package org.unicode.cldr.draft;

import java.text.ParseException;
import java.text.ParsePosition;

/**
 * @author markdavis
 */
public interface Parser<T, S extends CharSequence> {
    /**
     * Parses text from the beginning of the given string to produce an object. The method may not
     * use the entire text of the given string.
     *
     * @param source A <code>String</code> whose beginning should be parsed.
     * @return An <code>Object</code> parsed from the string.
     * @exception ParseException if the beginning of the specified string cannot be parsed.
     */
    public T parseObject(S source) throws ParseException;

    /**
     * Parses text from a string to produce an object.
     *
     * <p>The method attempts to parse text starting at the index given by <code>pos</code>. If
     * parsing succeeds, then the index of <code>pos</code> is updated to the index after the last
     * character used (parsing does not necessarily use all characters up to the end of the string),
     * and the parsed object is returned. The updated <code>pos</code> can be used to indicate the
     * starting point for the next call to this method. If an error occurs, then the index of <code>
     * pos</code> is not changed, the error index of <code>pos</code> is set to the index of the
     * character where the error occurred, and null is returned.
     *
     * @param source A <code>String</code>, part of which should be parsed.
     * @param pos A <code>ParsePosition</code> object with index and error index information as
     *     described above.
     * @return An <code>Object</code> parsed from the string. In case of error, returns null.
     * @exception NullPointerException if <code>pos</code> is null.
     */
    public T parseObject(S source, ParsePosition pos);
}
