#!/usr/bin/env python
# copyright 2002-2010 LOGILAB S.A. (Paris, FRANCE), all rights reserved.
# contact http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This file is part of logilab-constraint.
#
# logilab-constraint is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by the
# Free Software Foundation, either version 2.1 of the License, or (at your
# option) any later version.
#
# logilab-constraint is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
# for more details.
#
# You should have received a copy of the GNU Lesser General Public License along
# with logilab-constraint. If not, see <http://www.gnu.org/licenses/>.
"""N queens problem
The problem is solved with a EnumeratorDistributor splitting
the smallest domain in at most N subdomains."""


from logilab.constraint import fd, Repository, Solver
from logilab.constraint.distributors import EnumeratorDistributor


def rooks(size=8, verbose=0):
    variables = []
    domains = {}
    constraints = []
    for i in range(size):
        name = "R%02d" % i
        variables.append(name)
        domains[name] = fd.FiniteDomain(range(size))

    for r1 in range(size):
        for r2 in range(size):
            q1 = "R%02d" % r1
            q2 = "R%02d" % r2
            if r1 < r2:
                D = {"q1": q1, "q2": q2, "r1": r1, "r2": r2}
                c = fd.make_expression((q1, q2), f"{D['q1']} != {D['q2']}")
                constraints.append(c)

    r = Repository(variables, domains, constraints)
    yield from Solver(EnumeratorDistributor()).solve_all(r, verbose)


def draw_solution(s):
    size = len(s)
    board = ""
    queens = s.items()
    queens.sort()
    board += "_" * (size * 2 + 1) + "\n"
    for i in range(size):
        qj = queens[i][1]
        for j in range(size):
            if j != qj:
                board += "|" + "·-"[(i + j) % 2]
            else:
                board += "|R"
        board += "|\n"
    board += "¯" * (size * 2 + 1)
    print(board)


def main(args=None):
    import sys
    import getopt

    if args is None:
        args = sys.argv[1:]
    opts, args = getopt.getopt(args, "dvf")
    display = 0
    verbose = 0
    first = 0
    if args:
        size = int(args[0])
    else:
        size = 8
    for o, v in opts:
        if o == "-d":
            display = 1
        elif o == "-v":
            verbose += 1
        elif o == "-f":
            first = 1
    count = 0
    for sol in rooks(size, verbose):
        count += 1
        if display:
            print("solution #%d" % count)
            draw_solution(sol)
            print("*" * 80)
        if first:
            break
    if not display:
        print("Use -d option to display solutions")
    print(count, "solutions found.")


if __name__ == "__main__":
    #     import hotshot
    #     p = hotshot.Profile('/tmp/queens.prof')
    # p.runcall(main)
    # p.close()
    main()
