/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
/** Used to generate unique IDs for events. */
let uniqueIds = 0;
/**
 * Creates a browser MouseEvent with the specified options.
 * @docs-private
 */
export function createMouseEvent(type, clientX = 0, clientY = 0, offsetX = 1, offsetY = 1, button = 0, modifiers = {}) {
    // Note: We cannot determine the position of the mouse event based on the screen
    // because the dimensions and position of the browser window are not available
    // To provide reasonable `screenX` and `screenY` coordinates, we simply use the
    // client coordinates as if the browser is opened in fullscreen.
    const screenX = clientX;
    const screenY = clientY;
    const event = new MouseEvent(type, {
        bubbles: true,
        cancelable: true,
        composed: true,
        view: window,
        detail: 0,
        relatedTarget: null,
        screenX,
        screenY,
        clientX,
        clientY,
        ctrlKey: modifiers.control,
        altKey: modifiers.alt,
        shiftKey: modifiers.shift,
        metaKey: modifiers.meta,
        button: button,
        buttons: 1,
    });
    // The `MouseEvent` constructor doesn't allow us to pass these properties into the constructor.
    // Override them to `1`, because they're used for fake screen reader event detection.
    if (offsetX != null) {
        defineReadonlyEventProperty(event, 'offsetX', offsetX);
    }
    if (offsetY != null) {
        defineReadonlyEventProperty(event, 'offsetY', offsetY);
    }
    return event;
}
/**
 * Creates a browser `PointerEvent` with the specified options. Pointer events
 * by default will appear as if they are the primary pointer of their type.
 * https://www.w3.org/TR/pointerevents2/#dom-pointerevent-isprimary.
 *
 * For example, if pointer events for a multi-touch interaction are created, the non-primary
 * pointer touches would need to be represented by non-primary pointer events.
 *
 * @docs-private
 */
export function createPointerEvent(type, clientX = 0, clientY = 0, offsetX, offsetY, options = { isPrimary: true }) {
    const event = new PointerEvent(type, {
        bubbles: true,
        cancelable: true,
        composed: true,
        view: window,
        clientX,
        clientY,
        ...options,
    });
    if (offsetX != null) {
        defineReadonlyEventProperty(event, 'offsetX', offsetX);
    }
    if (offsetY != null) {
        defineReadonlyEventProperty(event, 'offsetY', offsetY);
    }
    return event;
}
/**
 * Creates a browser TouchEvent with the specified pointer coordinates.
 * @docs-private
 */
export function createTouchEvent(type, pageX = 0, pageY = 0, clientX = 0, clientY = 0) {
    // We cannot use the `TouchEvent` or `Touch` because Firefox and Safari lack support.
    // TODO: Switch to the constructor API when it is available for Firefox and Safari.
    const event = document.createEvent('UIEvent');
    const touchDetails = { pageX, pageY, clientX, clientY, identifier: uniqueIds++ };
    // TS3.6 removes the initUIEvent method and suggests porting to "new UIEvent()".
    event.initUIEvent(type, true, true, window, 0);
    // Most of the browsers don't have a "initTouchEvent" method that can be used to define
    // the touch details.
    defineReadonlyEventProperty(event, 'touches', [touchDetails]);
    defineReadonlyEventProperty(event, 'targetTouches', [touchDetails]);
    defineReadonlyEventProperty(event, 'changedTouches', [touchDetails]);
    return event;
}
/**
 * Creates a keyboard event with the specified key and modifiers.
 * @docs-private
 */
export function createKeyboardEvent(type, keyCode = 0, key = '', modifiers = {}) {
    return new KeyboardEvent(type, {
        bubbles: true,
        cancelable: true,
        composed: true,
        view: window,
        keyCode: keyCode,
        key: key,
        shiftKey: modifiers.shift,
        metaKey: modifiers.meta,
        altKey: modifiers.alt,
        ctrlKey: modifiers.control,
    });
}
/**
 * Creates a fake event object with any desired event type.
 * @docs-private
 */
export function createFakeEvent(type, bubbles = false, cancelable = true, composed = true) {
    return new Event(type, { bubbles, cancelable, composed });
}
/**
 * Defines a readonly property on the given event object. Readonly properties on an event object
 * are always set as configurable as that matches default readonly properties for DOM event objects.
 */
function defineReadonlyEventProperty(event, propertyName, value) {
    Object.defineProperty(event, propertyName, { get: () => value, configurable: true });
}
//# sourceMappingURL=data:application/json;base64,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