/**
 *    Copyright (C) 2023-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#include "mongo/db/auth/authorization_session.h"
#include "mongo/db/commands.h"
#include "mongo/db/repl/replication_coordinator.h"
#include "mongo/db/s/config/sharding_catalog_manager.h"
#include "mongo/db/s/sharding_ready.h"
#include "mongo/db/s/transition_to_sharded_cluster_gen.h"
#include "mongo/s/sharding_state.h"

#define MONGO_LOGV2_DEFAULT_COMPONENT ::mongo::logv2::LogComponent::kSharding

namespace mongo {
namespace {

class TransitionToShardedClusterCommand : public TypedCommand<TransitionToShardedClusterCommand> {
public:
    using Request = TransitionToShardedCluster;

    std::string help() const override {
        return "Transitions a replica set into a sharded cluster.";
    }

    AllowedOnSecondary secondaryAllowed(ServiceContext*) const override {
        return AllowedOnSecondary::kNever;
    }

    bool adminOnly() const override {
        return true;
    }

    class Invocation final : public InvocationBase {
    public:
        using InvocationBase::InvocationBase;

        void typedRun(OperationContext* opCtx) {
            CommandHelpers::uassertCommandRunWithMajority(Request::kCommandName,
                                                          opCtx->getWriteConcern());

            uassert(8002900,
                    "Command can only be run on replica sets",
                    repl::ReplicationCoordinator::get(opCtx)->getSettings().isReplSet());

            uassert(8002901,
                    "Command cannot be run on dedicated shards",
                    serverGlobalParams.clusterRole.has(ClusterRole::ConfigServer));

            uassert(8263000,
                    "Command was attempted when the replica set is still in the process of being "
                    "upgraded or downgraded (upgradeBackCompat or downgradeBackCompat options are "
                    "on). Please finish upgrading or downgrading the replica set before attempting "
                    "to convert it to a sharded cluster.",
                    !serverGlobalParams.upgradeBackCompat &&
                        !serverGlobalParams.downgradeBackCompat);

            if (!ShardingState::get(opCtx)->enabled()) {
                ShardingCatalogManager::get(opCtx)->installConfigShardIdentityDocument(opCtx);
            }

            auto shardingReady = ShardingReady::get(opCtx);
            shardingReady->scheduleTransitionToConfigShard(opCtx);
            shardingReady->waitUntilReady(opCtx);
        }

    private:
        NamespaceString ns() const override {
            return {};
        }

        bool supportsWriteConcern() const override {
            return true;
        }

        void doCheckAuthorization(OperationContext* opCtx) const override {
            uassert(ErrorCodes::Unauthorized,
                    "Unauthorized",
                    AuthorizationSession::get(opCtx->getClient())
                        ->isAuthorizedForActionsOnResource(
                            ResourcePattern::forClusterResource(request().getDbName().tenantId()),
                            ActionType::internal));
        }
    };
};
MONGO_REGISTER_COMMAND(TransitionToShardedClusterCommand).forShard();

}  // namespace
}  // namespace mongo
