// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Collections.Generic;
using System.Linq;
using System.Security.Authentication;
using Xunit;

namespace System.Net.Security.Tests
{
    public class TlsFrameHelperTests
    {
        [Fact]
        public void SniHelper_ValidData_Ok()
        {
            InvalidClientHello(s_validClientHello, -1, shouldPass: true);
        }

        [Theory]
        [MemberData(nameof(InvalidClientHelloData))]
        public void SniHelper_InvalidData_Fails(int id, byte[] clientHello)
        {
            InvalidClientHello(clientHello, id, shouldPass: false);
        }

        [Theory]
        [MemberData(nameof(InvalidClientHelloDataTruncatedBytes))]
        public void SniHelper_TruncatedData_Fails(int id, byte[] clientHello)
        {
            InvalidClientHello(clientHello, id, shouldPass: false);
        }

        private void InvalidClientHello(byte[] clientHello, int id, bool shouldPass)
        {
            string ret = TlsFrameHelper.GetServerName(clientHello);
            if (shouldPass)
                Assert.NotNull(ret);
            else
                Assert.Null(ret);
        }

        [Fact]
        public void TlsFrameHelper_ValidData_Ok()
        {
            TlsFrameHelper.TlsFrameInfo info = default;
            Assert.True(TlsFrameHelper.TryGetFrameInfo(s_validClientHello, ref info));

            Assert.Equal(SslProtocols.Tls12, info.Header.Version);
            Assert.Equal(208, info.Header.Length);
            Assert.Equal(SslProtocols.Tls12, info.SupportedVersions);
            Assert.Equal(TlsFrameHelper.ApplicationProtocolInfo.None, info.ApplicationProtocols);
        }

        [Fact]
        public void TlsFrameHelper_Tls12ClientHello_Ok()
        {
            TlsFrameHelper.TlsFrameInfo info = default;
            Assert.True(TlsFrameHelper.TryGetFrameInfo(s_Tls12ClientHello, ref info));

#pragma warning disable SYSLIB0039
            Assert.Equal(SslProtocols.Tls, info.Header.Version);
            Assert.Equal(SslProtocols.Tls | SslProtocols.Tls12, info.SupportedVersions);
#pragma warning restore SYSLIB0039
            Assert.Equal(TlsFrameHelper.ApplicationProtocolInfo.Http11 | TlsFrameHelper.ApplicationProtocolInfo.Http2, info.ApplicationProtocols);
        }

        [Fact]
        public void TlsFrameHelper_Tls13ClientHello_Ok()
        {
            TlsFrameHelper.TlsFrameInfo info = default;
            Assert.True(TlsFrameHelper.TryGetFrameInfo(s_Tls13ClientHello, ref info));

#pragma warning disable SYSLIB0039
            Assert.Equal(SslProtocols.Tls, info.Header.Version);
            Assert.Equal(SslProtocols.Tls | SslProtocols.Tls11 | SslProtocols.Tls12 | SslProtocols.Tls13, info.SupportedVersions);
#pragma warning restore SYSLIB0039
            Assert.Equal(TlsFrameHelper.ApplicationProtocolInfo.Other, info.ApplicationProtocols);
        }

        [Fact]
        public void TlsFrameHelper_Tls12ServerHello_Ok()
        {
            TlsFrameHelper.TlsFrameInfo info = default;
            Assert.True(TlsFrameHelper.TryGetFrameInfo(s_Tls12ServerHello, ref info));

            Assert.Equal(SslProtocols.Tls12, info.Header.Version);
            Assert.Equal(SslProtocols.Tls12, info.SupportedVersions);
            Assert.Equal(TlsFrameHelper.ApplicationProtocolInfo.Http2, info.ApplicationProtocols);
        }

        [Fact]
        public void TlsFrameHelper_UnifiedClientHello_Ok()
        {
            TlsFrameHelper.TlsFrameInfo info = default;
            Assert.True(TlsFrameHelper.TryGetFrameHeader(s_UnifiedHello,  ref info.Header));
            Assert.Equal(75, info.Header.Length);

            Assert.True(TlsFrameHelper.TryGetFrameInfo(s_UnifiedHello, ref info));
#pragma warning disable CS0618      // Ssl2 and Ssl3 are obsolete
#pragma warning disable SYSLIB0039  // Tls is obsolete
            Assert.Equal(SslProtocols.Ssl2, info.Header.Version);
            Assert.Equal(SslProtocols.Ssl2 | SslProtocols.Tls, info.SupportedVersions);
#pragma warning restore CS0618
#pragma warning restore SYSLIB0039
            Assert.Equal(TlsContentType.Handshake, info.Header.Type);
            Assert.Equal(TlsFrameHelper.ApplicationProtocolInfo.None, info.ApplicationProtocols);
            Assert.Equal(TlsHandshakeType.ClientHello, info.HandshakeType);
        }

        [Fact]
        public void TlsFrameHelper_TlsClientHelloNoExtensions_Ok()
        {
            TlsFrameHelper.TlsFrameInfo info = default;
            Assert.True(TlsFrameHelper.TryGetFrameInfo(s_TlsClientHelloNoExtensions, ref info));
            Assert.Equal(SslProtocols.Tls12, info.Header.Version);
            Assert.Equal(SslProtocols.Tls12, info.SupportedVersions);
            Assert.Equal(TlsContentType.Handshake, info.Header.Type);
            Assert.Equal(TlsFrameHelper.ApplicationProtocolInfo.None, info.ApplicationProtocols);
            Assert.Equal(TlsHandshakeType.ClientHello, info.HandshakeType);
        }

        public static IEnumerable<object[]> InvalidClientHelloData()
        {
            int id = 0;
            foreach (byte[] invalidClientHello in InvalidClientHello())
            {
                id++;
                yield return new object[] { id, invalidClientHello };
            }
        }

        public static IEnumerable<object[]> InvalidClientHelloDataTruncatedBytes()
        {
            // converting to base64 first to remove duplicated test cases
            var uniqueInvalidHellos = new HashSet<string>();
            foreach (byte[] invalidClientHello in InvalidClientHello())
            {
                for (int i = 0; i < invalidClientHello.Length; i++)
                {
                    uniqueInvalidHellos.Add(Convert.ToBase64String(invalidClientHello.Take(i).ToArray()));
                }
            }

            for (int i = 0; i < s_validClientHello.Length; i++)
            {
                uniqueInvalidHellos.Add(Convert.ToBase64String(s_validClientHello.Take(i).ToArray()));
            }

            int id = 0;
            foreach (string invalidClientHello in uniqueInvalidHellos)
            {
                id++;
                yield return new object[] { id, Convert.FromBase64String(invalidClientHello) };
            }
        }

        private static byte[] s_validClientHello = new byte[] {
            // SslPlainText.(ContentType+ProtocolVersion)
            0x16, 0x03, 0x03,
            // SslPlainText.length
            0x00, 0xCB,
            // Handshake.msg_type (client hello)
            0x01,
            // Handshake.length
            0x00, 0x00, 0xC7,
            // ClientHello.client_version
            0x03, 0x03,
            // ClientHello.random
            0x0C, 0x3C, 0x85, 0x78, 0xCA,
            0x67, 0x70, 0xAA, 0x38, 0xCB,
            0x28, 0xBC, 0xDC, 0x3E, 0x30,
            0xBF, 0x11, 0x96, 0x95, 0x1A,
            0xB9, 0xF0, 0x99, 0xA4, 0x91,
            0x09, 0x13, 0xB4, 0x89, 0x94,
            0x27, 0x2E,
            // ClientHello.SessionId
            0x00,
            // ClientHello.cipher_suites
            0x00, 0x2A, 0xC0, 0x2C, 0xC0,
            0x2B, 0xC0, 0x30, 0xC0, 0x2F,
            0x00, 0x9F, 0x00, 0x9E, 0xC0,
            0x24, 0xC0, 0x23, 0xC0, 0x28,
            0xC0, 0x27, 0xC0, 0x0A, 0xC0,
            0x09, 0xC0, 0x14, 0xC0, 0x13,
            0x00, 0x9D, 0x00, 0x9C, 0x00,
            0x3D, 0x00, 0x3C, 0x00, 0x35,
            0x00, 0x2F, 0x00, 0x0A,
            // ClientHello.compression_methods
            0x01, 0x01,
            // ClientHello.extension_list_length
            0x00, 0x74,
            // Extension.extension_type (server_name)
            0x00, 0x00,
            // ServerNameListExtension.length
            0x00, 0x39,
            // ServerName.length
            0x00, 0x37,
            // ServerName.type
            0x00,
            // HostName.length
            0x00, 0x34,
            // HostName.bytes
            0x61, 0x61, 0x61, 0x61, 0x61,
            0x61, 0x61, 0x61, 0x61, 0x61,
            0x61, 0x61, 0x61, 0x61, 0x61,
            0x61, 0x61, 0x61, 0x61, 0x61,
            0x61, 0x61, 0x61, 0x61, 0x61,
            0x61, 0x61, 0x61, 0x61, 0x61,
            0x61, 0x61, 0x61, 0x61, 0x61,
            0x61, 0x61, 0x61, 0x61, 0x61,
            0x61, 0x61, 0x61, 0x61, 0x61,
            0x61, 0x61, 0x61, 0x61, 0x61,
            0x61, 0x61,
            // Extension.extension_type (00 0A)
            0x00, 0x0A,
            // Extension 0A
            0x00, 0x08, 0x00, 0x06, 0x00,
            0x1D, 0x00, 0x17, 0x00, 0x18,
            // Extension.extension_type (00 0B)
            0x00, 0x0B,
            // Extension 0B
            0x00, 0x02, 0x01, 0x00,
            // Extension.extension_type (00 0D)
            0x00, 0x0D,
            // Extension 0D
            0x00, 0x14, 0x00, 0x12, 0x04,
            0x01, 0x05, 0x01, 0x02, 0x01,
            0x04, 0x03, 0x05, 0x03, 0x02,
            0x03, 0x02, 0x02, 0x06, 0x01,
            0x06, 0x03,
            // Extension.extension_type (00 23)
            0x00, 0x23,
            // Extension 00 23
            0x00, 0x00,
            // Extension.extension_type (00 17)
            0x00, 0x17,
            // Extension 17
            0x00, 0x00,
            // Extension.extension_type (FF 01)
            0xFF, 0x01,
            // Extension FF01
            0x00, 0x01, 0x00
        };

        private static byte[] s_Tls12ClientHello = new byte[] {
            // SslPlainText.(ContentType+ProtocolVersion)
            0x16, 0x03, 0x01,
            // SslPlainText.length
            0x00, 0xD1,
            // Handshake.msg_type (client hello)
            0x01,
            // Handshake.length
            0x00, 0x00, 0xCD,
            // ClientHello.client_version
            0x03, 0x03,
            // ClientHello.random
            0x0C, 0x3C, 0x85, 0x78, 0xCA,
            0x67, 0x70, 0xAA, 0x38, 0xCB,
            0x28, 0xBC, 0xDC, 0x3E, 0x30,
            0xBF, 0x11, 0x96, 0x95, 0x1A,
            0xB9, 0xF0, 0x99, 0xA4, 0x91,
            0x09, 0x13, 0xB4, 0x89, 0x94,
            0x27, 0x2E,
            // ClientHello.SessionId
            0x00,
            // ClientHello.cipher_suites_length
            0x00, 0x5C,
            // ClientHello.cipher_suites
            0xC0, 0x30, 0xC0, 0x2C, 0xC0, 0x28, 0xC0, 0x24,
            0xC0, 0x14, 0xC0, 0x0A, 0x00, 0x9f, 0x00, 0x6B,
            0x00, 0x39, 0xCC, 0xA9, 0xCC, 0xA8, 0xCC, 0xAA,
            0xFF, 0x85, 0x00, 0xC4, 0x00, 0x88, 0x00, 0x81,
            0x00, 0x9D, 0x00, 0x3D, 0x00, 0x35, 0x00, 0xC0,
            0x00, 0x84, 0xC0, 0x2f, 0xC0, 0x2B, 0xC0, 0x27,
            0xC0, 0x23, 0xC0, 0x13, 0xC0, 0x09, 0x00, 0x9E,
            0x00, 0x67, 0x00, 0x33, 0x00, 0xBE, 0x00, 0x45,
            0x00, 0x9C, 0x00, 0x3C, 0x00, 0x2F, 0x00, 0xBA,
            0x00, 0x41, 0xC0, 0x11, 0xC0, 0x07, 0x00, 0x05,
            0x00, 0x04, 0xC0, 0x12, 0xC0, 0x08, 0x00, 0x16,
            0x00, 0x0a, 0x00, 0xff,
            // ClientHello.compression_methods
            0x01, 0x01,
            // ClientHello.extension_list_length
            0x00, 0x48,
            // Extension.extension_type (ec_point_formats)
            0x00, 0x0b, 0x00, 0x02, 0x01, 0x00,
            // Extension.extension_type (supported_groups)
            0x00, 0x0A, 0x00, 0x08, 0x00, 0x06, 0x00, 0x1D,
            0x00, 0x17, 0x00, 0x18,
            // Extension.extension_type (session_ticket)
            0x00, 0x23, 0x00, 0x00,
            // Extension.extension_type (signature_algorithms)
            0x00, 0x0D, 0x00, 0x1C, 0x00, 0x1A, 0x06, 0x01,
            0x06, 0x03, 0xEF, 0xEF, 0x05, 0x01, 0x05, 0x03,
            0x04, 0x01, 0x04, 0x03, 0xEE, 0xEE, 0xED, 0xED,
            0x03, 0x01, 0x03, 0x03, 0x02, 0x01, 0x02, 0x03,
            // Extension.extension_type (application_level_Protocol)
            0x00, 0x10, 0x00, 0x0e, 0x00, 0x0C, 0x02, 0x68,
            0x32, 0x08, 0x68, 0x74, 0x74, 0x70, 0x2F, 0x31,
            0x2E, 0x31
        };

        private static byte[] s_Tls13ClientHello = new byte[] {
            // SslPlainText.(ContentType+ProtocolVersion)
            0x16, 0x03, 0x01,
            // SslPlainText.length
            0x01, 0x08,
            // Handshake.msg_type (client hello)
            0x01,
            // Handshake.length
            0x00, 0x01, 0x04,
            // ClientHello.client_version
            0x03, 0x03,
            // ClientHello.random
            0x0C, 0x3C, 0x85, 0x78, 0xCA, 0x67, 0x70, 0xAA,
            0x38, 0xCB, 0x28, 0xBC, 0xDC, 0x3E, 0x30, 0xBF,
            0x11, 0x96, 0x95, 0x1A, 0xB9, 0xF0, 0x99, 0xA4,
            0x91, 0x09, 0x13, 0xB4, 0x89, 0x94, 0x27, 0x2E,
            // ClientHello.SessionId_Length
            0x20,
            // ClientHello.SessionId
            0x0C, 0x3C, 0x85, 0x78, 0xCA, 0x67, 0x70, 0xAA,
            0x38, 0xCB, 0x28, 0xBC, 0xDC, 0x3E, 0x30, 0xBF,
            0x11, 0x96, 0x95, 0x1A, 0xB9, 0xF0, 0x99, 0xA4,
            0x91, 0x09, 0x13, 0xB4, 0x89, 0x94, 0x27, 0x2E,
            // ClientHello.cipher_suites_length
            0x00, 0x0C,
            // ClientHello.cipher_suites
            0x13, 0x02, 0x13, 0x03, 0x13, 0x01, 0xC0, 0x14,
            0xc0, 0x30, 0x00, 0xFF,
            // ClientHello.compression_methods
            0x01, 0x00,
            // ClientHello.extension_list_length
            0x00, 0xAF,
            // Extension.extension_type (server_name) (10.211.55.2)
            0x00, 0x00, 0x00, 0x10, 0x00, 0x0e, 0x00, 0x00,
            0x0B, 0x31, 0x30, 0x2E, 0x32, 0x31, 0x31, 0x2E,
            0x35, 0x35, 0x2E, 0x32,
            // Extension.extension_type (ec_point_formats)
            0x00, 0x0B, 0x00, 0x04, 0x03, 0x00, 0x01, 0x02,
            // Extension.extension_type (supported_groups)
            0x00, 0x0A, 0x00, 0x0C, 0x00, 0x0A, 0x00, 0x1D,
            0x00, 0x17, 0x00, 0x1E, 0x00, 0x19, 0x00, 0x18,
            // Extension.extension_type (application_level_Protocol) (boo)
            0x00, 0x10, 0x00, 0x06, 0x00, 0x04, 0x03, 0x62,
            0x6f, 0x6f,
            // Extension.extension_type (encrypt_then_mac)
            0x00, 0x16, 0x00, 0x00,
            // Extension.extension_type (extended_master_key_secret)
            0x00, 0x17, 0x00, 0x00,
            // Extension.extension_type (signature_algorithms)
            0x00, 0x0D, 0x00, 0x30, 0x00, 0x2E,
            0x06, 0x03, 0xEF, 0xEF, 0x05, 0x01, 0x05, 0x03,
            0x06, 0x03, 0xEF, 0xEF, 0x05, 0x01, 0x05, 0x03,
            0x06, 0x03, 0xEF, 0xEF, 0x05, 0x01, 0x05, 0x03,
            0x04, 0x01, 0x04, 0x03, 0xEE, 0xEE, 0xED, 0xED,
            0x03, 0x01, 0x03, 0x03, 0x02, 0x01, 0x02, 0x03,
            0x03, 0x01, 0x03, 0x03, 0x02, 0x01,
            // Extension.extension_type (supported_versions)
            0x00, 0x2B, 0x00, 0x09, 0x08, 0x03, 0x04, 0x03,
            0x03, 0x03, 0x02, 0x03, 0x01,
            // Extension.extension_type (psk_key_exchange_modes)
            0x00, 0x2D, 0x00, 0x02, 0x01, 0x01,
            // Extension.extension_type (key_share)
            0x00, 0x33, 0x00, 0x26, 0x00, 0x24, 0x00, 0x1D,
            0x00, 0x20,
            0x04, 0x01, 0x04, 0x03, 0xEE, 0xEE, 0xED, 0xED,
            0x03, 0x01, 0x03, 0x03, 0x02, 0x01, 0x02, 0x03,
            0x04, 0x01, 0x04, 0x03, 0xEE, 0xEE, 0xED, 0xED,
            0x03, 0x01, 0x03, 0x03, 0x02, 0x01, 0x02, 0x03
        };

        private static byte[] s_Tls12ServerHello = new byte[] {
            // SslPlainText.(ContentType+ProtocolVersion)
            0x16, 0x03, 0x03,
            // SslPlainText.length
            0x00, 0x64,
            // Handshake.msg_type (srever hello)
            0x02,
            // Handshake.length
            0x00, 0x00, 0x60,
            // ServerHello.client_version
            0x03, 0x03,
            // ServerHello.random
            0x0C, 0x3C, 0x85, 0x78, 0xCA,
            0x67, 0x70, 0xAA, 0x38, 0xCB,
            0x28, 0xBC, 0xDC, 0x3E, 0x30,
            0xBF, 0x11, 0x96, 0x95, 0x1A,
            0xB9, 0xF0, 0x99, 0xA4, 0x91,
            0x09, 0x13, 0xB4, 0x89, 0x94,
            0x27, 0x2E,
            // ServerHello.SessionId_Length
            0x20,
            // ServerHello.SessionId
            0x0C, 0x3C, 0x85, 0x78, 0xCA, 0x67, 0x70, 0xAA,
            0x38, 0xCB, 0x28, 0xBC, 0xDC, 0x3E, 0x30, 0xBF,
            0x11, 0x96, 0x95, 0x1A, 0xB9, 0xF0, 0x99, 0xA4,
            0x91, 0x09, 0x13, 0xB4, 0x89, 0x94, 0x27, 0x2E,
            // ServerHello.cipher_suite
            0xC0, 0x2B,
            // ServerHello.compression_method
            0x00,
            // ClientHello.extension_list_length
            0x00, 0x18,
            // Extension.extension_type (extended_master_secreet)
            0x00, 0x17, 0x00, 0x00,
            // Extension.extension_type (renegotiation_info)
            0xFF, 0x01, 0x00, 0x01, 0x00,
            // Extension.extension_type (ec_point_formats)
            0x00, 0x0B, 0x00, 0x02, 0x01, 0x00,
            // Extension.extension_type (application_level_Protocol)
            0x00, 0x10, 0x00, 0x05, 0x00, 0x03, 0x02, 0x68, 0x32,
        };

        internal static byte[] s_UnifiedHello = new byte[]
        {
            // Length
            0x80, 0x49,
            // ClientHello
            0x01,
            // Version
            0x03, 0x01,
            0x00, 0x30, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
            0x2F, 0x00, 0x00, 0x35, 0x00, 0x00, 0x04, 0x00,
            0x00, 0x05, 0x00, 0x00, 0x0A, 0x01, 0x00, 0x80,
            0x07, 0x00, 0xC0, 0x03, 0x00, 0x80, 0x00, 0x00,
            0x09, 0x06, 0x00, 0x40, 0x00, 0x00, 0x64, 0x00,
            0x00, 0x62, 0x00, 0x00, 0x03, 0x00, 0x00, 0x06,
            0x02, 0x00, 0x80, 0x04, 0x00, 0x80, 0x5B, 0x0B,
            0xA1, 0xEB, 0xBF, 0x2D, 0x57, 0xF5, 0xD1, 0x0F,
            0x52, 0x3B, 0x12, 0x9C, 0xF8, 0xD4,
        };

        private static byte[] s_TlsClientHelloNoExtensions = new byte[] {
            0x16, 0x03, 0x03, 0x00, 0x39, 0x01, 0x00, 0x00,
            0x35, 0x03, 0x03, 0x62, 0x5d, 0x50, 0x2a, 0x41,
            0x2f, 0xd8, 0xc3, 0x65, 0x35, 0xea, 0x01, 0x70,
            0x03, 0x7e, 0x7e, 0x2d, 0xd4, 0xfe, 0x93, 0x39,
            0xa4, 0x04, 0x66, 0xbb, 0x46, 0x91, 0x41, 0xc3,
            0x48, 0x87, 0x3d, 0x00, 0x00, 0x0e, 0x00, 0x3d,
            0x00, 0x3c, 0x00, 0x0a, 0x00, 0x35, 0x00, 0x2f,
            0x00, 0x05, 0x00, 0x04, 0x01, 0x00
        };

        private static IEnumerable<byte[]> InvalidClientHello()
        {
            // This test covers following test cases:
            // - Length of structure off by 1 (search for "length off by 1")
            // - Length of structure is max length (search for "max length")
            // - Type is invalid or unknown (i.e. SslPlainText.ClientType is not 0x16 - search for "unknown")
            // - Invalid utf-8 characters
            // in each case sni will be null or will cause parsing error - we only expect some parsing errors,
            // anything else is considered a bug
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01 - length off by 1
                0x00, 0x02, 0x00
            };

            // #2
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01 - max length
                0xFF, 0xFF, 0x00
            };

            // #3
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17 - length off by 1
                0x00, 0x01,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #4
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17 - max length
                0xFF, 0xFF,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #5
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23 - length off by 1
                0x00, 0x01,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #6
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23 - max length
                0xFF, 0xFF,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #7
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D - length off by 1
                0x00, 0x15, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #8
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D - max length
                0xFF, 0xFF, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #9
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B - length off by 1
                0x00, 0x03, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #10
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B - max length
                0xFF, 0xFF, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #11
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A - length off by 1
                0x00, 0x09, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #10
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A - max length
                0xFF, 0xFF, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #13
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length - length off by 1
                0x00, 0x35,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #14
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length - max length
                0xFF, 0xFF,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #15
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type - unknown
                0x01,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #16
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length - length off by 1
                0x00, 0x38,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #17
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length - max length
                0xFF, 0xFF,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #18
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length - length off by 1
                0x00, 0x3A,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #19
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length - max length
                0xFF, 0xFF,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #20
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name) - unknown
                0x01, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #21
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length - length off by 1
                0x00, 0x75,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #22
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length - max length
                0xFF, 0xFF,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #23
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods - length off by 1
                0x02, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #24
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods - max length
                0xFF, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #25
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites - length off by 1
                0x00, 0x2B, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #26
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites - max length
                0xFF, 0xFF, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #27
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId - length off by 1
                0x01,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #28
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId - max length
                0xFF,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #29
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length - length off by 1
                0x00, 0x00, 0xC8,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #30
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length - max length
                0xFF, 0xFF, 0xFF,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #31
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello) - unknown
                0x00,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #32
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length - length off by 1
                0x00, 0xCC,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #33
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length - max length
                0xFF, 0xFF,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #34
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion) - unknown
                0x01, 0x03, 0x04,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x58, 0xAA, 0x5F, 0xE7, 0x22,
                0xCF, 0x9F, 0x59, 0x8A, 0xC5,
                0x8B, 0x87, 0xC7, 0x62, 0x32,
                0x98, 0xD4, 0xD8, 0xA2, 0xBE,
                0x77, 0xCE, 0xA9, 0xCE, 0x42,
                0x25, 0x5A, 0x8B, 0xEE, 0x16,
                0x80, 0xF1,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };

            // #35
            yield return new byte[] {
                // SslPlainText.(ContentType+ProtocolVersion)
                0x16, 0x03, 0x03,
                // SslPlainText.length
                0x00, 0xCB,
                // Handshake.msg_type (client hello)
                0x01,
                // Handshake.length
                0x00, 0x00, 0xC7,
                // ClientHello.client_version
                0x03, 0x03,
                // ClientHello.random
                0x0C, 0x3C, 0x85, 0x78, 0xCA,
                0x67, 0x70, 0xAA, 0x38, 0xCB,
                0x28, 0xBC, 0xDC, 0x3E, 0x30,
                0xBF, 0x11, 0x96, 0x95, 0x1A,
                0xB9, 0xF0, 0x99, 0xA4, 0x91,
                0x09, 0x13, 0xB4, 0x89, 0x94,
                0x27, 0x2E,
                // ClientHello.SessionId
                0x00,
                // ClientHello.cipher_suites
                0x00, 0x2A, 0xC0, 0x2C, 0xC0,
                0x2B, 0xC0, 0x30, 0xC0, 0x2F,
                0x00, 0x9F, 0x00, 0x9E, 0xC0,
                0x24, 0xC0, 0x23, 0xC0, 0x28,
                0xC0, 0x27, 0xC0, 0x0A, 0xC0,
                0x09, 0xC0, 0x14, 0xC0, 0x13,
                0x00, 0x9D, 0x00, 0x9C, 0x00,
                0x3D, 0x00, 0x3C, 0x00, 0x35,
                0x00, 0x2F, 0x00, 0x0A,
                // ClientHello.compression_methods
                0x01, 0x01,
                // ClientHello.extension_list_length
                0x00, 0x74,
                // Extension.extension_type (server_name)
                0x00, 0x00,
                // ServerNameListExtension.length
                0x00, 0x39,
                // ServerName.length
                0x00, 0x37,
                // ServerName.type
                0x00,
                // HostName.length
                0x00, 0x34,
                // HostName.bytes
                0x80, 0x80, 0x80, 0x80, 0x61, // 0x80 0x80 0x80 0x80 is a forbidden utf-8 sequence
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61, 0x61, 0x61, 0x61,
                0x61, 0x61,
                // Extension.extension_type (00 0A)
                0x00, 0x0A,
                // Extension 0A
                0x00, 0x08, 0x00, 0x06, 0x00,
                0x1D, 0x00, 0x17, 0x00, 0x18,
                // Extension.extension_type (00 0B)
                0x00, 0x0B,
                // Extension 0B
                0x00, 0x02, 0x01, 0x00,
                // Extension.extension_type (00 0D)
                0x00, 0x0D,
                // Extension 0D
                0x00, 0x14, 0x00, 0x12, 0x04,
                0x01, 0x05, 0x01, 0x02, 0x01,
                0x04, 0x03, 0x05, 0x03, 0x02,
                0x03, 0x02, 0x02, 0x06, 0x01,
                0x06, 0x03,
                // Extension.extension_type (00 23)
                0x00, 0x23,
                // Extension 00 23
                0x00, 0x00,
                // Extension.extension_type (00 17)
                0x00, 0x17,
                // Extension 17
                0x00, 0x00,
                // Extension.extension_type (FF 01)
                0xFF, 0x01,
                // Extension FF01
                0x00, 0x01, 0x00
            };
        }
    }
}
