﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cleanroomsml/CleanRoomsMLRequest.h>
#include <aws/cleanroomsml/CleanRoomsML_EXPORTS.h>
#include <aws/cleanroomsml/model/PrivacyConfiguration.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CleanRoomsML {
namespace Model {

/**
 */
class CreateConfiguredModelAlgorithmAssociationRequest : public CleanRoomsMLRequest {
 public:
  AWS_CLEANROOMSML_API CreateConfiguredModelAlgorithmAssociationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateConfiguredModelAlgorithmAssociation"; }

  AWS_CLEANROOMSML_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The membership ID of the member who is associating this configured model
   * algorithm.</p>
   */
  inline const Aws::String& GetMembershipIdentifier() const { return m_membershipIdentifier; }
  inline bool MembershipIdentifierHasBeenSet() const { return m_membershipIdentifierHasBeenSet; }
  template <typename MembershipIdentifierT = Aws::String>
  void SetMembershipIdentifier(MembershipIdentifierT&& value) {
    m_membershipIdentifierHasBeenSet = true;
    m_membershipIdentifier = std::forward<MembershipIdentifierT>(value);
  }
  template <typename MembershipIdentifierT = Aws::String>
  CreateConfiguredModelAlgorithmAssociationRequest& WithMembershipIdentifier(MembershipIdentifierT&& value) {
    SetMembershipIdentifier(std::forward<MembershipIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the configured model algorithm that you
   * want to associate.</p>
   */
  inline const Aws::String& GetConfiguredModelAlgorithmArn() const { return m_configuredModelAlgorithmArn; }
  inline bool ConfiguredModelAlgorithmArnHasBeenSet() const { return m_configuredModelAlgorithmArnHasBeenSet; }
  template <typename ConfiguredModelAlgorithmArnT = Aws::String>
  void SetConfiguredModelAlgorithmArn(ConfiguredModelAlgorithmArnT&& value) {
    m_configuredModelAlgorithmArnHasBeenSet = true;
    m_configuredModelAlgorithmArn = std::forward<ConfiguredModelAlgorithmArnT>(value);
  }
  template <typename ConfiguredModelAlgorithmArnT = Aws::String>
  CreateConfiguredModelAlgorithmAssociationRequest& WithConfiguredModelAlgorithmArn(ConfiguredModelAlgorithmArnT&& value) {
    SetConfiguredModelAlgorithmArn(std::forward<ConfiguredModelAlgorithmArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the configured model algorithm association.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateConfiguredModelAlgorithmAssociationRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the configured model algorithm association.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateConfiguredModelAlgorithmAssociationRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the privacy configuration information for the configured model
   * algorithm association. This information includes the maximum data size that can
   * be exported.</p>
   */
  inline const PrivacyConfiguration& GetPrivacyConfiguration() const { return m_privacyConfiguration; }
  inline bool PrivacyConfigurationHasBeenSet() const { return m_privacyConfigurationHasBeenSet; }
  template <typename PrivacyConfigurationT = PrivacyConfiguration>
  void SetPrivacyConfiguration(PrivacyConfigurationT&& value) {
    m_privacyConfigurationHasBeenSet = true;
    m_privacyConfiguration = std::forward<PrivacyConfigurationT>(value);
  }
  template <typename PrivacyConfigurationT = PrivacyConfiguration>
  CreateConfiguredModelAlgorithmAssociationRequest& WithPrivacyConfiguration(PrivacyConfigurationT&& value) {
    SetPrivacyConfiguration(std::forward<PrivacyConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The optional metadata that you apply to the resource to help you categorize
   * and organize them. Each tag consists of a key and an optional value, both of
   * which you define.</p> <p>The following basic restrictions apply to tags:</p>
   * <ul> <li> <p>Maximum number of tags per resource - 50.</p> </li> <li> <p>For
   * each resource, each tag key must be unique, and each tag key can have only one
   * value.</p> </li> <li> <p>Maximum key length - 128 Unicode characters in
   * UTF-8.</p> </li> <li> <p>Maximum value length - 256 Unicode characters in
   * UTF-8.</p> </li> <li> <p>If your tagging schema is used across multiple services
   * and resources, remember that other services may have restrictions on allowed
   * characters. Generally allowed characters are: letters, numbers, and spaces
   * representable in UTF-8, and the following characters: + - = . _ : / @.</p> </li>
   * <li> <p>Tag keys and values are case sensitive.</p> </li> <li> <p>Do not use
   * aws:, AWS:, or any upper or lowercase combination of such as a prefix for keys
   * as it is reserved for AWS use. You cannot edit or delete tag keys with this
   * prefix. Values can have this prefix. If a tag value has aws as its prefix but
   * the key does not, then Clean Rooms ML considers it to be a user tag and will
   * count against the limit of 50 tags. Tags with only the key prefix of aws do not
   * count against your tags per resource limit.</p> </li> </ul>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateConfiguredModelAlgorithmAssociationRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateConfiguredModelAlgorithmAssociationRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_membershipIdentifier;
  bool m_membershipIdentifierHasBeenSet = false;

  Aws::String m_configuredModelAlgorithmArn;
  bool m_configuredModelAlgorithmArnHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  PrivacyConfiguration m_privacyConfiguration;
  bool m_privacyConfigurationHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CleanRoomsML
}  // namespace Aws
