//go:build fips

package fips

import (
	"slices"

	"golang.org/x/crypto/ssh"
)

// fipsKexAlgos specifies FIPS approved key-exchange algorithms implemented
// by this package.
var fipsKexAlgos = []string{ssh.KeyExchangeECDHP256, ssh.KeyExchangeECDHP384}

// fipsCiphers specifies FIPS approved cipher algorithms implemented by this
// package.
var fipsCiphers = []string{
	ssh.CipherAES128GCM, ssh.CipherAES256GCM,
	ssh.CipherAES128CTR, ssh.CipherAES192CTR, ssh.CipherAES256CTR,
}

// fipsMACs specifies FIPS approved MAC algorithms implemented by this
// package.
var fipsMACs = []string{
	ssh.HMACSHA256ETM, ssh.HMACSHA512ETM,
	ssh.HMACSHA256, ssh.HMACSHA512,
}

// fipsHostKeyAlgos specifies FIPS approved host-key algorithms implemented
// by this package.
var fipsHostKeyAlgos = []string{
	// Permitted based on NIST SP 800-186: https://csrc.nist.gov/csrc/media/Projects/cryptographic-module-validation-program/documents/fips%20140-3/FIPS%20140-3%20IG.pdf
	ssh.CertAlgoED25519v01,
	ssh.CertAlgoRSASHA256v01, ssh.CertAlgoRSASHA512v01,
	ssh.CertAlgoECDSA256v01, ssh.CertAlgoECDSA384v01,
	ssh.KeyAlgoECDSA256, ssh.KeyAlgoECDSA384,
	ssh.KeyAlgoRSASHA256, ssh.KeyAlgoRSASHA512,
}

// fipsPubKeyAuthAlgos specifies FIPS approved public key authentication algorithms.
var fipsPubKeyAuthAlgos = []string{
	// Permitted based on NIST SP 800-186: https://csrc.nist.gov/csrc/media/Projects/cryptographic-module-validation-program/documents/fips%20140-3/FIPS%20140-3%20IG.pdf
	ssh.KeyAlgoED25519,
	ssh.KeyAlgoRSASHA256, ssh.KeyAlgoRSASHA512,
	ssh.KeyAlgoECDSA256, ssh.KeyAlgoECDSA384,
}

type Algorithms struct {
	Ciphers        []string
	MACs           []string
	KeyExchanges   []string
	HostKeys       []string
	PublicKeyAuths []string
}

// SupportedAlgorithms returns algorithms currently implemented by this package,
// excluding those with security issues, which are returned by
// InsecureAlgorithms. The algorithms listed here are in preference order.
// FIPS 140-2 docs: https://csrc.nist.gov/pubs/fips/140-2/upd2/final
func SupportedAlgorithms() ssh.Algorithms {
	return ssh.Algorithms{
		Ciphers:        slices.Clone(fipsCiphers),
		MACs:           slices.Clone(fipsMACs),
		KeyExchanges:   slices.Clone(fipsKexAlgos),
		HostKeys:       slices.Clone(fipsHostKeyAlgos),
		PublicKeyAuths: slices.Clone(fipsPubKeyAuthAlgos),
	}
}

// DefaultAlgorithms returns algorithms currently implemented by this package.
// For FIPS, ED25519 key exchanges are not supported and will result in a panic.
// The algorithms listed here are in preference order.
func DefaultAlgorithms() ssh.Algorithms {
	config := &ssh.ServerConfig{}
	config.SetDefaults()

	// Filter ED25519 key exchange algorithms
	config.KeyExchanges = slices.DeleteFunc(config.KeyExchanges, func(s string) bool {
		return s == ssh.KeyExchangeMLKEM768X25519 ||
			s == ssh.KeyExchangeCurve25519 ||
			// This alias is added by x/crypto/ssh automatically
			s == "curve25519-sha256@libssh.org"
	})

	return ssh.Algorithms{
		Ciphers:      config.Ciphers,
		MACs:         config.MACs,
		KeyExchanges: config.KeyExchanges,
		// Leave HostKeys empty because the algorithms are derived from the host keys
		HostKeys:       nil,
		PublicKeyAuths: config.PublicKeyAuthAlgorithms,
	}
}
