#!/bin/sh
# busybox udhcp setup script
# shellcheck shell=sh disable=SC1008

PATH=/bin:/usr/bin:/sbin:/usr/sbin

peer_var()
{
	[ -n "$1" ] && [ "$1" != "yes" ]
}

update_dns()
{
	peer_var "${PEER_DNS}" && return
	[ -z "${domain}" ] && [ -z "${dns}" ] && return

	conf="# Generated by udhcpc for ${interface}\n"
	[ -n "${domain}" ] && conf="${conf}domain ${domain}\n"
	for i in ${dns} ; do
		conf="${conf}nameserver ${i}\n"
	done
	if command -v resolvconf >/dev/null; then
		printf "${conf}" | resolvconf -a ${interface}
	else
		printf "${conf}" > /etc/resolv.conf
		chmod 644 /etc/resolv.conf
	fi
}

update_ntp()
{
	peer_var "${PEER_NTP}" && return
	[ -z "${ntpsrv}" ] && return

	conf="# Generated by udhcpc for interface ${interface}\n"
	conf="${conf}restrict default noquery notrust nomodify\n"
	conf="${conf}restrict 127.0.0.1\n"
	for i in ${ntpsrv} ; do
		conf="${conf}restrict ${i} nomodify notrap noquery\n"
		conf="${conf}server ${i}\n"
	done
	conf="${conf}driftfile /var/lib/ntp/ntp.drift\n"
	conf="${conf}logfile /var/log/ntp.log\n"
	printf "${conf}" > /etc/ntp.conf
	chmod 644 /etc/ntp.conf
}

update_hostname()
{
	peer_var "${PEER_HOSTNAME}" && return
	[ -z "${hostname}" ] && return

	myhost="$(hostname)"
	[ -z "${myhost}" ] || [ "${myhost}" = "(none)" ] && hostname "${hostname}"
}

update_interface()
{
	[ -n "${broadcast}" ] && broadcast="broadcast ${broadcast}"
	[ -n "${subnet}" ] && netmask="netmask ${subnet}"
	[ -n "${mtu}" ] && mtu="mtu ${mtu}"
	ifconfig "${interface}" ${ip} ${broadcast} ${netmask} ${mtu}
}

update_classless_routes()
{
	if [ -n "${staticroutes}" ] ; then
		max_routes=128
		metric=
		[ -n "${IF_METRIC}" ] && metric="metric ${IF_METRIC}"
		while [ -n "$1" ] && [ -n "$2" ] && [ $max_routes -gt 0 ]; do
			gw_arg=
			if [ "$2" != '0.0.0.0' ]; then
				gw_arg="gw $2"
			fi

			[ ${1##*/} -eq 32 ] && type=host || type=net
			route add -$type "$1" ${gw_arg} ${metric} dev "${interface}"
			max=$((max-1))
			shift 2
		done
	fi
}
update_routes()
{
	peer_var "${PEER_ROUTERS}" && return

	# RFC 3442
	[ -n "${staticroutes}" ] && update_classless_routes $staticroutes

	# If the DHCP server returns both a Classless Static Routes option and
	# a Router option, the DHCP client MUST ignore the Router option.
	if [ -n "${router}" ] && [ -z "${staticroutes}" ] ; then
		metric=
		[ -n "${IF_METRIC}" ] && metric="metric ${IF_METRIC}"
		for i in ${router} ; do
			route add default gw "${i}" ${metric} dev "${interface}"
		done
	fi
}

deconfig()
{
	ifconfig "${interface}" 0.0.0.0

	if ! peer_var "${PEER_ROUTERS}" ; then
		while route del default dev "${interface}" >/dev/null 2>&1; do
			:
		done
	fi

	if ! peer_var "${PEER_DNS}" ; then
		command -v resolvconf >/dev/null && resolvconf -d "${interface}"
	fi
}

if [ -r "/run/udhcpc-${interface}.conf" ]; then
	# shellcheck disable=SC1090
	. "/run/udhcpc-${interface}.conf"
fi

case "$1" in
	bound|renew)
		update_hostname
		update_interface
		update_routes
		update_dns
		update_ntp
		;;
	deconfig|leasefail)
		deconfig
		;;
	nak)
		echo "nak: ${message}"
		;;
	*)
		echo "unknown option $1" >&2
		echo "Usage: $0 {bound|deconfig|leasefail|nak|renew}" >&2
		exit 1
		;;
esac

exit 0
