/*
  This file is part of TALER
  (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
  details.

  You should have received a copy of the GNU Affero General Public License
  along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file merchant/backend/taler-merchant-httpd_private-get-reports.c
 * @brief implementation of GET /private/reports
 * @author Christian Grothoff
 */
#include "platform.h"
#include "taler-merchant-httpd_private-get-reports.h"
#include <taler/taler_json_lib.h>
#include <taler/taler_dbevents.h>


/**
 * Sensible bound on the limit.
 */
#define MAX_DELTA 1024


/**
 * Callback for listing reports.
 *
 * @param cls closure with a `json_t *`
 * @param report_id unique identifier of the report
 * @param report_description human-readable description
 * @param frequency how often the report is generated
 */
static void
add_report (void *cls,
            uint64_t report_id,
            const char *report_description,
            struct GNUNET_TIME_Relative frequency)
{
  json_t *reports = cls;
  json_t *entry;

  entry = GNUNET_JSON_PACK (
    GNUNET_JSON_pack_uint64 ("report_serial",
                             report_id),
    GNUNET_JSON_pack_string ("description",
                             report_description),
    GNUNET_JSON_pack_time_rel ("report_frequency",
                               frequency));
  GNUNET_assert (NULL != entry);
  GNUNET_assert (0 ==
                 json_array_append_new (reports,
                                        entry));
}


MHD_RESULT
TMH_private_get_reports (const struct TMH_RequestHandler *rh,
                         struct MHD_Connection *connection,
                         struct TMH_HandlerContext *hc)
{
  int64_t limit = -20;
  uint64_t offset;
  enum GNUNET_DB_QueryStatus qs;
  json_t *reports;

  (void) rh;
  TALER_MHD_parse_request_snumber (connection,
                                   "limit",
                                   &limit);
  if (limit > 0)
    offset = 0;
  else
    offset = INT64_MAX;
  TALER_MHD_parse_request_number (connection,
                                  "offset",
                                  &offset);
  if ( (-MAX_DELTA > limit) ||
       (limit > MAX_DELTA) )
  {
    GNUNET_break_op (0);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_BAD_REQUEST,
                                       TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                       "limit");
  }

  reports = json_array ();
  GNUNET_assert (NULL != reports);
  qs = TMH_db->select_reports (TMH_db->cls,
                               hc->instance->settings.id,
                               limit,
                               offset,
                               &add_report,
                               reports);
  if (qs < 0)
  {
    json_decref (reports);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_INTERNAL_SERVER_ERROR,
                                       TALER_EC_GENERIC_DB_FETCH_FAILED,
                                       "select_reports");
  }

  return TALER_MHD_REPLY_JSON_PACK (
    connection,
    MHD_HTTP_OK,
    GNUNET_JSON_pack_array_steal ("reports",
                                  reports));
}
