"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const node_process_1 = __importDefault(require("node:process"));
const electron_better_ipc_1 = require("electron-better-ipc");
const electron_util_1 = require("electron-util");
const elementReady = require("element-ready");
const remote_1 = require("@electron/remote");
const selectors_1 = __importDefault(require("./browser/selectors"));
const autoplay_1 = require("./autoplay");
const conversation_list_1 = require("./browser/conversation-list");
async function withMenu(menuButtonElement, callback) {
    const { classList } = document.documentElement;
    // Prevent the dropdown menu from displaying
    classList.add('hide-dropdowns');
    // Click the menu button
    menuButtonElement.click();
    // Wait for the menu to close before removing the 'hide-dropdowns' class
    const menuLayer = document.querySelector('.j83agx80.cbu4d94t.l9j0dhe7.jgljxmt5.be9z9djy > div:nth-child(2) > div');
    if (menuLayer) {
        const observer = new MutationObserver(() => {
            if (!menuLayer.hasChildNodes()) {
                classList.remove('hide-dropdowns');
                observer.disconnect();
            }
        });
        observer.observe(menuLayer, { childList: true });
    }
    else {
        // Fallback in case .uiContextualLayerPositioner is missing
        classList.remove('hide-dropdowns');
    }
    await callback();
}
async function isNewSidebar() {
    // TODO: stopOnDomReady might not be needed
    await elementReady('[role=navigation] > div > div', { stopOnDomReady: false });
    const sidebars = document.querySelectorAll('[role=navigation] > div > div');
    return sidebars.length === 2;
}
async function withSettingsMenu(callback) {
    const newSidebar = await isNewSidebar();
    // Wait for navigation pane buttons to show up
    await elementReady(newSidebar ? selectors_1.default.userMenuNewSidebar : selectors_1.default.userMenu, { stopOnDomReady: false });
    const settingsMenu = newSidebar
        ? document.querySelectorAll(selectors_1.default.userMenuNewSidebar)[0]
        : document.querySelector(selectors_1.default.userMenu);
    await withMenu(settingsMenu, callback);
}
async function selectMenuItem(itemNumber) {
    let selector;
    // Wait for menu to show up
    await elementReady(selectors_1.default.conversationMenuSelectorNewDesign, { stopOnDomReady: false });
    const items = document.querySelectorAll(`${selectors_1.default.conversationMenuSelectorNewDesign} [role=menuitem]`);
    // Negative items will select from the end
    if (itemNumber < 0) {
        selector = -itemNumber <= items.length ? items[items.length + itemNumber] : null;
    }
    else {
        selector = itemNumber <= items.length ? items[itemNumber - 1] : null;
    }
    if (selector) {
        selector.click();
    }
}
async function selectOtherListViews(itemNumber) {
    // In case one of other views is shown
    clickBackButton();
    const newSidebar = await isNewSidebar();
    if (newSidebar) {
        const items = document.querySelectorAll(`${selectors_1.default.viewsMenu} > span > a`);
        const selector = itemNumber <= items.length ? items[itemNumber - 1] : null;
        console.log(selector);
        if (selector) {
            selector.click();
        }
    }
    else {
        await withSettingsMenu(() => {
            selectMenuItem(itemNumber);
        });
    }
}
function clickBackButton() {
    const backButton = document.querySelector('._30yy._2oc9');
    if (backButton) {
        backButton.click();
    }
}
electron_better_ipc_1.ipcRenderer.answerMain('show-preferences', async () => {
    if (isPreferencesOpen()) {
        return;
    }
    await openPreferences();
});
electron_better_ipc_1.ipcRenderer.answerMain('new-conversation', async () => {
    document.querySelector('[href="/new/"]').click();
});
electron_better_ipc_1.ipcRenderer.answerMain('new-room', async () => {
    document.querySelector('.x16n37ib .x1i10hfl.x6umtig.x1b1mbwd.xaqea5y.xav7gou.x1ypdohk.xe8uvvx.xdj266r.x11i5rnm.xat24cr.x1mh8g0r.x16tdsg8.x1hl2dhg.xggy1nq.x87ps6o.x1lku1pv.x1a2a7pz.x6s0dn4.x14yjl9h.xudhj91.x18nykt9.xww2gxu.x972fbf.xcfux6l.x1qhh985.xm0m39n.x9f619.x78zum5.xl56j7k.xexx8yu.x4uap5.x18d9i69.xkhd6sd.x1n2onr6.xc9qbxq.x14qfxbe.x1qhmfi1').click();
});
electron_better_ipc_1.ipcRenderer.answerMain('log-out', async () => {
    const useWorkChat = await electron_better_ipc_1.ipcRenderer.callMain('get-config-useWorkChat');
    if (useWorkChat) {
        document.querySelector('._5lxs._3qct._p').click();
        // Menu creation is slow
        setTimeout(() => {
            const nodes = document.querySelectorAll('._54nq._9jo._558b._2n_z li:last-child a');
            nodes[nodes.length - 1].click();
        }, 250);
    }
    else {
        await withSettingsMenu(() => {
            selectMenuItem(-1);
        });
    }
});
electron_better_ipc_1.ipcRenderer.answerMain('find', () => {
    document.querySelector('[type="search"]').focus();
});
async function openSearchInConversation() {
    const mainView = document.querySelector('.rq0escxv.l9j0dhe7.du4w35lb.j83agx80.rj1gh0hx.buofh1pr.g5gj957u.hpfvmrgz.i1fnvgqd.gs1a9yip.owycx6da.btwxx1t3.jb3vyjys.gitj76qy');
    const rightSidebarIsClosed = Boolean(mainView.querySelector('div:only-child'));
    if (rightSidebarIsClosed) {
        document.documentElement.classList.add('hide-r-sidebar');
        document.querySelector('.j9ispegn.pmk7jnqg.k4urcfbm.datstx6m.b5wmifdl.kr520xx4.mdpwds66.b2cqd1jy.n13yt9zj.eh67sqbx')?.click();
    }
    await elementReady(selectors_1.default.rightSidebarSegments, { stopOnDomReady: false });
    const segments = document.querySelectorAll(selectors_1.default.rightSidebarSegments).length;
    // If there are three segmetns in right sidebar (two users chat) then button index is 4
    // If there are not three segments (usually four, it's a group chat) then button index is 6
    const buttonIndex = segments === 3 ? 4 : 6;
    await elementReady(selectors_1.default.rightSidebarButtons, { stopOnDomReady: false });
    const buttonList = document.querySelectorAll(selectors_1.default.rightSidebarButtons);
    if (buttonList.length > buttonIndex) {
        buttonList[buttonIndex].click();
    }
    // If right sidebar was closed when shortcut was clicked, then close it back.
    if (rightSidebarIsClosed) {
        document.querySelector('.j9ispegn.pmk7jnqg.k4urcfbm.datstx6m.b5wmifdl.kr520xx4.mdpwds66.b2cqd1jy.n13yt9zj.eh67sqbx')?.click();
        // Observe sidebar so when it's hidden, remove the utility class. This prevents split
        // display of sidebar.
        const sidebarObserver = new MutationObserver(records => {
            const removedRecords = records.filter(({ removedNodes }) => removedNodes.length > 0 && removedNodes[0].tagName === 'DIV');
            // In case there is a div removed, hide utility class and stop observing
            if (removedRecords.length > 0) {
                document.documentElement.classList.remove('hide-r-sidebar');
                sidebarObserver.disconnect();
            }
        });
        sidebarObserver.observe(mainView, { childList: true, subtree: true });
    }
}
electron_better_ipc_1.ipcRenderer.answerMain('search', () => {
    openSearchInConversation();
});
electron_better_ipc_1.ipcRenderer.answerMain('insert-gif', () => {
    document.querySelector('.x1n2onr6.x1iyjqo2.xw2csxc > div:nth-child(3) > span > div').click();
});
electron_better_ipc_1.ipcRenderer.answerMain('insert-emoji', async () => {
    document.querySelector('.x1n2onr6.x1iyjqo2.xw2csxc > div:nth-child(5) > span > div').click();
});
electron_better_ipc_1.ipcRenderer.answerMain('insert-sticker', () => {
    document.querySelector('.x1n2onr6.x1iyjqo2.xw2csxc > div:nth-child(2) > span > div').click();
});
electron_better_ipc_1.ipcRenderer.answerMain('attach-files', () => {
    document.querySelector('.x1n2onr6.x1iyjqo2.xw2csxc > div:nth-child(1) > span > div').click();
});
electron_better_ipc_1.ipcRenderer.answerMain('focus-text-input', () => {
    document.querySelector('[role=textbox][contenteditable=true]').focus();
});
electron_better_ipc_1.ipcRenderer.answerMain('next-conversation', nextConversation);
electron_better_ipc_1.ipcRenderer.answerMain('previous-conversation', previousConversation);
electron_better_ipc_1.ipcRenderer.answerMain('mute-conversation', async () => {
    await openMuteModal();
});
electron_better_ipc_1.ipcRenderer.answerMain('delete-conversation', async () => {
    await deleteSelectedConversation();
});
electron_better_ipc_1.ipcRenderer.answerMain('hide-conversation', async () => {
    const index = selectedConversationIndex();
    if (index !== -1) {
        await hideSelectedConversation();
        const key = index + 1;
        await jumpToConversation(key);
    }
});
async function openHiddenPreferences() {
    if (!isPreferencesOpen()) {
        document.documentElement.classList.add('hide-preferences-window');
        const style = document.createElement('style');
        // Hide both the backdrop and the preferences dialog
        style.textContent = `${selectors_1.default.preferencesSelector} ._3ixn, ${selectors_1.default.preferencesSelector} ._59s7 { opacity: 0 !important }`;
        document.body.append(style);
        await openPreferences();
        return true;
    }
    return false;
}
async function toggleSounds({ checked }) {
    const shouldClosePreferences = await openHiddenPreferences();
    const soundsCheckbox = document.querySelector(`${selectors_1.default.preferencesSelector} ${selectors_1.default.messengerSoundsSelector}`);
    if (typeof checked === 'undefined' || checked !== soundsCheckbox.checked) {
        soundsCheckbox.click();
    }
    if (shouldClosePreferences) {
        await closePreferences();
    }
}
electron_better_ipc_1.ipcRenderer.answerMain('toggle-sounds', toggleSounds);
electron_better_ipc_1.ipcRenderer.answerMain('toggle-mute-notifications', async () => {
    const shouldClosePreferences = await openHiddenPreferences();
    const notificationCheckbox = document.querySelector(selectors_1.default.notificationCheckbox);
    if (shouldClosePreferences) {
        await closePreferences();
    }
    // TODO: Fix notifications
    if (notificationCheckbox === null) {
        return false;
    }
    return !notificationCheckbox.checked;
});
electron_better_ipc_1.ipcRenderer.answerMain('toggle-message-buttons', async () => {
    const showMessageButtons = await electron_better_ipc_1.ipcRenderer.callMain('get-config-showMessageButtons');
    document.body.classList.toggle('show-message-buttons', !showMessageButtons);
});
electron_better_ipc_1.ipcRenderer.answerMain('show-chats-view', async () => {
    await selectOtherListViews(1);
});
electron_better_ipc_1.ipcRenderer.answerMain('show-people-view', async () => {
    await selectOtherListViews(2);
});
electron_better_ipc_1.ipcRenderer.answerMain('show-marketplace-view', async () => {
    await selectOtherListViews(3);
});
electron_better_ipc_1.ipcRenderer.answerMain('show-requests-view', async () => {
    await selectOtherListViews(4);
});
electron_better_ipc_1.ipcRenderer.answerMain('show-archive-view', async () => {
    await selectOtherListViews(5);
});
electron_better_ipc_1.ipcRenderer.answerMain('toggle-video-autoplay', () => {
    (0, autoplay_1.toggleVideoAutoplay)();
});
electron_better_ipc_1.ipcRenderer.answerMain('reload', () => {
    location.reload();
});
async function setTheme() {
    const theme = await electron_better_ipc_1.ipcRenderer.callMain('get-config-theme');
    remote_1.nativeTheme.themeSource = theme;
    setThemeElement(document.documentElement);
    updateVibrancy();
}
function setThemeElement(element) {
    const useDarkColors = Boolean(remote_1.nativeTheme.shouldUseDarkColors);
    element.classList.toggle('dark-mode', useDarkColors);
    element.classList.toggle('light-mode', !useDarkColors);
    element.classList.toggle('__fb-dark-mode', useDarkColors);
    element.classList.toggle('__fb-light-mode', !useDarkColors);
    removeThemeClasses(useDarkColors);
}
function removeThemeClasses(useDarkColors) {
    // TODO: Workaround for Facebooks buggy frontend
    // The ui sometimes hardcodes ligth mode classes in the ui. This removes them so the class
    // in the root element would be used.
    const className = useDarkColors ? '__fb-light-mode' : '__fb-dark-mode';
    for (const element of document.querySelectorAll(`.${className}`)) {
        element.classList.remove(className);
    }
}
async function observeTheme() {
    /* Main document's class list */
    const observer = new MutationObserver((records) => {
        // Find records that had class attribute changed
        const classRecords = records.filter(record => record.type === 'attributes' && record.attributeName === 'class');
        // Check if dark mode classes exists
        const isDark = classRecords.some(record => {
            const { classList } = record.target;
            return classList.contains('dark-mode') && classList.contains('__fb-dark-mode');
        });
        // If config and class list don't match, update class list
        if (remote_1.nativeTheme.shouldUseDarkColors !== isDark) {
            setTheme();
        }
    });
    observer.observe(document.documentElement, { attributes: true, attributeFilter: ['class'] });
    /* Added nodes (dialogs, etc.) */
    const observerNew = new MutationObserver((records) => {
        const nodeRecords = records.filter(record => record.addedNodes.length > 0);
        for (const nodeRecord of nodeRecords) {
            for (const newNode of nodeRecord.addedNodes) {
                const { classList } = newNode;
                const isLight = classList.contains('light-mode') || classList.contains('__fb-light-mode');
                if (remote_1.nativeTheme.shouldUseDarkColors === isLight) {
                    setThemeElement(newNode);
                }
            }
        }
    });
    /* Observe only elements where new nodes may need dark mode */
    const menuElements = await elementReady('.j83agx80.cbu4d94t.l9j0dhe7.jgljxmt5.be9z9djy > div:nth-of-type(2) > div', { stopOnDomReady: false });
    if (menuElements) {
        observerNew.observe(menuElements, { childList: true });
    }
    // Attribute notation needed here to guarantee exact (not partial) match.
    const modalElements = await elementReady(selectors_1.default.preferencesSelector, { stopOnDomReady: false });
    if (modalElements) {
        observerNew.observe(modalElements, { childList: true });
    }
}
async function setPrivateMode() {
    const privateMode = await electron_better_ipc_1.ipcRenderer.callMain('get-config-privateMode');
    document.documentElement.classList.toggle('private-mode', privateMode);
    if (electron_util_1.is.macos) {
        (0, conversation_list_1.sendConversationList)();
    }
}
async function updateVibrancy() {
    const { classList } = document.documentElement;
    classList.remove('sidebar-vibrancy', 'full-vibrancy');
    const vibrancy = await electron_better_ipc_1.ipcRenderer.callMain('get-config-vibrancy');
    switch (vibrancy) {
        case 'sidebar':
            classList.add('sidebar-vibrancy');
            break;
        case 'full':
            classList.add('full-vibrancy');
            break;
        default:
    }
    electron_better_ipc_1.ipcRenderer.callMain('set-vibrancy');
}
async function updateSidebar() {
    const { classList } = document.documentElement;
    classList.remove('sidebar-hidden', 'sidebar-force-narrow', 'sidebar-force-wide');
    const sidebar = await electron_better_ipc_1.ipcRenderer.callMain('get-config-sidebar');
    switch (sidebar) {
        case 'hidden':
            classList.add('sidebar-hidden');
            break;
        case 'narrow':
            classList.add('sidebar-force-narrow');
            break;
        case 'wide':
            classList.add('sidebar-force-wide');
            break;
        default:
    }
}
// TODO: Implement this function
async function updateDoNotDisturb() {
    const shouldClosePreferences = await openHiddenPreferences();
    if (shouldClosePreferences) {
        await closePreferences();
    }
}
function renderOverlayIcon(messageCount) {
    const canvas = document.createElement('canvas');
    canvas.height = 128;
    canvas.width = 128;
    canvas.style.letterSpacing = '-5px';
    const ctx = canvas.getContext('2d');
    ctx.fillStyle = '#f42020';
    ctx.beginPath();
    ctx.ellipse(64, 64, 64, 64, 0, 0, 2 * Math.PI);
    ctx.fill();
    ctx.textAlign = 'center';
    ctx.fillStyle = 'white';
    ctx.font = '90px sans-serif';
    ctx.fillText(String(Math.min(99, messageCount)), 64, 96);
    return canvas;
}
electron_better_ipc_1.ipcRenderer.answerMain('update-sidebar', () => {
    updateSidebar();
});
electron_better_ipc_1.ipcRenderer.answerMain('set-theme', setTheme);
electron_better_ipc_1.ipcRenderer.answerMain('set-private-mode', setPrivateMode);
electron_better_ipc_1.ipcRenderer.answerMain('update-vibrancy', () => {
    updateVibrancy();
});
electron_better_ipc_1.ipcRenderer.answerMain('render-overlay-icon', (messageCount) => ({
    data: renderOverlayIcon(messageCount).toDataURL(),
    text: String(messageCount),
}));
electron_better_ipc_1.ipcRenderer.answerMain('render-native-emoji', (emoji) => {
    const canvas = document.createElement('canvas');
    const context = canvas.getContext('2d');
    const systemFont = electron_util_1.is.linux ? 'emoji, system-ui' : 'system-ui';
    canvas.width = 256;
    canvas.height = 256;
    context.textAlign = 'center';
    context.textBaseline = 'middle';
    if (electron_util_1.is.macos) {
        context.font = `256px ${systemFont}`;
        context.fillText(emoji, 128, 154);
    }
    else {
        context.textBaseline = 'bottom';
        context.font = `225px ${systemFont}`;
        context.fillText(emoji, 128, 256);
    }
    const dataUrl = canvas.toDataURL();
    return dataUrl;
});
electron_better_ipc_1.ipcRenderer.answerMain('zoom-reset', async () => {
    await setZoom(1);
});
electron_better_ipc_1.ipcRenderer.answerMain('zoom-in', async () => {
    let zoomFactor = await electron_better_ipc_1.ipcRenderer.callMain('get-config-zoomFactor');
    zoomFactor += 0.1;
    if (zoomFactor < 1.6) {
        await setZoom(zoomFactor);
    }
});
electron_better_ipc_1.ipcRenderer.answerMain('zoom-out', async () => {
    let zoomFactor = await electron_better_ipc_1.ipcRenderer.callMain('get-config-zoomFactor');
    zoomFactor -= 0.1;
    if (zoomFactor >= 0.8) {
        await setZoom(zoomFactor);
    }
});
electron_better_ipc_1.ipcRenderer.answerMain('jump-to-conversation', async (key) => {
    await jumpToConversation(key);
});
async function nextConversation() {
    const index = selectedConversationIndex(1);
    if (index !== -1) {
        await selectConversation(index);
    }
}
async function previousConversation() {
    const index = selectedConversationIndex(-1);
    if (index !== -1) {
        await selectConversation(index);
    }
}
async function jumpToConversation(key) {
    const index = key - 1;
    await selectConversation(index);
}
// Focus on the conversation with the given index
async function selectConversation(index) {
    const list = await elementReady(selectors_1.default.conversationList, { stopOnDomReady: false });
    if (!list) {
        console.error('Could not find conversations list', selectors_1.default.conversationList);
        return;
    }
    const conversation = list.children[index];
    if (!conversation) {
        console.error('Could not find conversation', index);
        return;
    }
    conversation.querySelector('[role=link]').click();
}
function selectedConversationIndex(offset = 0) {
    const selected = document.querySelector(selectors_1.default.selectedConversation);
    if (!selected) {
        return -1;
    }
    const newSelected = selected.parentNode.parentNode.parentNode;
    const list = [...newSelected.parentNode.children];
    const index = list.indexOf(newSelected) + offset;
    return ((index % list.length) + list.length) % list.length;
}
async function setZoom(zoomFactor) {
    const node = document.querySelector('#zoomFactor');
    node.textContent = `${selectors_1.default.conversationSelector} {zoom: ${zoomFactor} !important}`;
    await electron_better_ipc_1.ipcRenderer.callMain('set-config-zoomFactor', zoomFactor);
}
async function withConversationMenu(callback) {
    // eslint-disable-next-line @typescript-eslint/ban-types
    let menuButton = null;
    const conversation = document.querySelector(`${selectors_1.default.selectedConversation}`)?.parentElement?.parentElement?.parentElement?.parentElement;
    menuButton = conversation?.querySelector('[aria-label=Menu][role=button]') ?? null;
    if (menuButton) {
        await withMenu(menuButton, callback);
    }
}
async function openMuteModal() {
    await withConversationMenu(() => {
        selectMenuItem(2);
    });
}
/*
This function assumes:
- There is a selected conversation.
- That the conversation already has its conversation menu open.

In other words, you should only use this function within a callback that is provided to `withConversationMenu()`, because `withConversationMenu()` makes sure to have the conversation menu open before executing the callback and closes the conversation menu afterwards.
*/
function isSelectedConversationGroup() {
    return Boolean(document.querySelector(`${selectors_1.default.conversationMenuSelectorNewDesign} [role=menuitem]:nth-child(4)`));
}
async function hideSelectedConversation() {
    await withConversationMenu(() => {
        const [isGroup, isNotGroup] = [5, 6];
        selectMenuItem(isSelectedConversationGroup() ? isGroup : isNotGroup);
    });
}
async function deleteSelectedConversation() {
    await withConversationMenu(() => {
        const [isGroup, isNotGroup] = [6, 7];
        selectMenuItem(isSelectedConversationGroup() ? isGroup : isNotGroup);
    });
}
async function openPreferences() {
    await withSettingsMenu(() => {
        selectMenuItem(1);
    });
    await elementReady(selectors_1.default.preferencesSelector, { stopOnDomReady: false });
}
function isPreferencesOpen() {
    return Boolean(document.querySelector('[aria-label=Preferences]'));
}
async function closePreferences() {
    // Wait for the preferences window to be closed, then remove the class from the document
    const preferencesOverlayObserver = new MutationObserver(records => {
        const removedRecords = records.filter(({ removedNodes }) => removedNodes.length > 0 && removedNodes[0].tagName === 'DIV');
        // In case there is a div removed, hide utility class and stop observing
        if (removedRecords.length > 0) {
            document.documentElement.classList.remove('hide-preferences-window');
            preferencesOverlayObserver.disconnect();
        }
    });
    const preferencesOverlay = document.querySelector(selectors_1.default.preferencesSelector);
    // Get the parent of preferences, that's not getting deleted
    const preferencesParent = preferencesOverlay.closest('div:not([class])');
    preferencesOverlayObserver.observe(preferencesParent, { childList: true });
    const closeButton = preferencesOverlay.querySelector(selectors_1.default.closePreferencesButton);
    closeButton?.click();
}
function insertionListener(event) {
    if (event.animationName === 'nodeInserted' && event.target) {
        event.target.dispatchEvent(new Event('mouseover', { bubbles: true }));
    }
}
async function observeAutoscroll() {
    const mainElement = await elementReady('._4sp8', { stopOnDomReady: false });
    if (!mainElement) {
        return;
    }
    const scrollToBottom = () => {
        // eslint-disable-next-line @typescript-eslint/ban-types
        const scrollableElement = document.querySelector('[role=presentation] .scrollable');
        if (scrollableElement) {
            scrollableElement.scroll({
                top: Number.MAX_SAFE_INTEGER,
                behavior: 'smooth',
            });
        }
    };
    const hookMessageObserver = async () => {
        const chatElement = await elementReady('[role=presentation] .scrollable [role = region] > div[id ^= "js_"]', { stopOnDomReady: false });
        if (chatElement) {
            // Scroll to the bottom when opening different conversation
            scrollToBottom();
            const messageObserver = new MutationObserver((record) => {
                const newMessages = record.filter(record => 
                // The mutation is an addition
                record.addedNodes.length > 0
                    // ... of a div       (skip the "seen" status change)
                    && record.addedNodes[0].tagName === 'DIV'
                    // ... on the last child       (skip previous messages added when scrolling up)
                    && chatElement.lastChild.contains(record.target));
                if (newMessages.length > 0) {
                    // Scroll to the bottom when there are new messages
                    scrollToBottom();
                }
            });
            messageObserver.observe(chatElement, { childList: true, subtree: true });
        }
    };
    hookMessageObserver();
    // Hook it again if conversation changes
    const conversationObserver = new MutationObserver(hookMessageObserver);
    conversationObserver.observe(mainElement, { childList: true });
}
async function observeThemeBugs() {
    const rootObserver = new MutationObserver((record) => {
        const newNodes = record
            .filter(record => record.addedNodes.length > 0 || record.removedNodes.length > 0);
        if (newNodes) {
            removeThemeClasses(Boolean(remote_1.nativeTheme.shouldUseDarkColors));
        }
    });
    rootObserver.observe(document.documentElement, { childList: true, subtree: true });
}
// Listen for emoji element dom insertion
document.addEventListener('animationstart', insertionListener, false);
// Inject a global style node to maintain custom appearance after conversation change or startup
document.addEventListener('DOMContentLoaded', async () => {
    const style = document.createElement('style');
    style.id = 'zoomFactor';
    document.body.append(style);
    // Set the zoom factor if it was set before quitting
    const zoomFactor = await electron_better_ipc_1.ipcRenderer.callMain('get-config-zoomFactor');
    setZoom(zoomFactor);
    // Enable OS specific styles
    document.documentElement.classList.add(`os-${node_process_1.default.platform}`);
    // Restore sidebar view state to what is was set before quitting
    updateSidebar();
    // Activate Dark Mode if it was set before quitting
    setTheme();
    // Observe for dark mode changes
    observeTheme();
    // Activate Private Mode if it was set before quitting
    setPrivateMode();
    // Configure do not disturb
    if (electron_util_1.is.macos) {
        await updateDoNotDisturb();
    }
    // Prevent flash of white on startup when in dark mode
    // TODO: find a CSS-only solution
    if (!electron_util_1.is.macos && remote_1.nativeTheme.shouldUseDarkColors) {
        document.documentElement.style.backgroundColor = '#1e1e1e';
    }
    // Disable autoplay if set in settings
    (0, autoplay_1.toggleVideoAutoplay)();
    // Hook auto-scroll observer
    observeAutoscroll();
    // Hook broken dark mode observer
    observeThemeBugs();
});
// Handle title bar double-click.
window.addEventListener('dblclick', (event) => {
    const target = event.target;
    const titleBar = target.closest('._36ic._5l-3,._5742,._6-xk,._673w');
    if (!titleBar) {
        return;
    }
    electron_better_ipc_1.ipcRenderer.callMain('titlebar-doubleclick');
}, {
    passive: true,
});
window.addEventListener('load', async () => {
    if (location.pathname.startsWith('/login')) {
        const keepMeSignedInCheckbox = document.querySelector('#u_0_0');
        const keepMeSignedInConfig = await electron_better_ipc_1.ipcRenderer.callMain('get-config-keepMeSignedIn');
        keepMeSignedInCheckbox.checked = keepMeSignedInConfig;
        keepMeSignedInCheckbox.addEventListener('change', async () => {
            const keepMeSignedIn = await electron_better_ipc_1.ipcRenderer.callMain('get-config-keepMeSignedIn');
            await electron_better_ipc_1.ipcRenderer.callMain('set-config-keepMeSignedIn', keepMeSignedIn);
        });
    }
});
// Toggles styles for inactive window
window.addEventListener('blur', () => {
    document.documentElement.classList.add('is-window-inactive');
});
window.addEventListener('focus', () => {
    document.documentElement.classList.remove('is-window-inactive');
});
// It's not possible to add multiple accelerators
// so this needs to be done the old-school way
document.addEventListener('keydown', async (event) => {
    // The `!event.altKey` part is a workaround for https://github.com/electron/electron/issues/13895
    const combineKey = electron_util_1.is.macos ? event.metaKey : event.ctrlKey && !event.altKey;
    if (!combineKey) {
        return;
    }
    if (event.key === ']') {
        await nextConversation();
    }
    if (event.key === '[') {
        await previousConversation();
    }
    const number = Number.parseInt(event.code.slice(-1), 10);
    if (number >= 1 && number <= 9) {
        await jumpToConversation(number);
    }
});
// Pass events sent via `window.postMessage` on to the main process
window.addEventListener('message', async ({ data: { type, data } }) => {
    if (type === 'notification') {
        showNotification(data);
    }
    if (type === 'notification-reply') {
        await sendReply(data.reply);
        if (data.previousConversation) {
            await selectConversation(data.previousConversation);
        }
    }
});
function showNotification({ id, title, body, icon, silent }) {
    const image = new Image();
    image.crossOrigin = 'anonymous';
    image.src = icon;
    image.addEventListener('load', () => {
        const canvas = document.createElement('canvas');
        const context = canvas.getContext('2d');
        canvas.width = image.width;
        canvas.height = image.height;
        context.drawImage(image, 0, 0, image.width, image.height);
        electron_better_ipc_1.ipcRenderer.callMain('notification', {
            id,
            title,
            body,
            icon: canvas.toDataURL(),
            silent,
        });
    });
}
async function sendReply(message) {
    const inputField = document.querySelector('[contenteditable="true"]');
    if (!inputField) {
        return;
    }
    const previousMessage = inputField.textContent;
    // Send message
    inputField.focus();
    insertMessageText(message, inputField);
    const sendButton = await elementReady('._30yy._38lh', { stopOnDomReady: false });
    if (!sendButton) {
        console.error('Could not find send button');
        return;
    }
    sendButton.click();
    // Restore (possible) previous message
    if (previousMessage) {
        insertMessageText(previousMessage, inputField);
    }
}
function insertMessageText(text, inputField) {
    // Workaround: insert placeholder value to get execCommand working
    if (!inputField.textContent) {
        const event = new InputEvent('textInput', {
            bubbles: true,
            cancelable: true,
            data: '_',
            view: window,
        });
        inputField.dispatchEvent(event);
    }
    document.execCommand('selectAll', false, undefined);
    document.execCommand('insertText', false, text);
}
electron_better_ipc_1.ipcRenderer.answerMain('notification-callback', (data) => {
    window.postMessage({ type: 'notification-callback', data }, '*');
});
electron_better_ipc_1.ipcRenderer.answerMain('notification-reply-callback', async (data) => {
    const previousConversation = selectedConversationIndex();
    data.previousConversation = previousConversation;
    window.postMessage({ type: 'notification-reply-callback', data }, '*');
});
