"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.sendConversationList = void 0;
const electron_better_ipc_1 = require("electron-better-ipc");
const element_ready_1 = __importDefault(require("element-ready"));
const lodash_1 = require("lodash");
const selectors_1 = __importDefault(require("./selectors"));
const icon = {
    read: 'data-caprine-icon',
    unread: 'data-caprine-icon-unread',
};
const padding = {
    top: 3,
    right: 0,
    bottom: 3,
    left: 0,
};
function drawIcon(size, img) {
    const canvas = document.createElement('canvas');
    if (img) {
        canvas.width = size + padding.left + padding.right;
        canvas.height = size + padding.top + padding.bottom;
        const ctx = canvas.getContext('2d');
        ctx.beginPath();
        ctx.arc((size / 2) + padding.left, (size / 2) + padding.top, (size / 2), 0, Math.PI * 2, true);
        ctx.closePath();
        ctx.clip();
        ctx.drawImage(img, padding.left, padding.top, size, size);
    }
    else {
        canvas.width = 0;
        canvas.height = 0;
    }
    return canvas;
}
// Return canvas with rounded image
async function urlToCanvas(url, size) {
    return new Promise(resolve => {
        const img = new Image();
        img.setAttribute('crossorigin', 'anonymous');
        img.addEventListener('load', () => {
            resolve(drawIcon(size, img));
        });
        img.addEventListener('error', () => {
            console.error('Image not found', url);
            resolve(drawIcon(size));
        });
        img.src = url;
    });
}
async function createIcons(element, url) {
    const canvas = await urlToCanvas(url, 50);
    element.setAttribute(icon.read, canvas.toDataURL());
    const markerSize = 8;
    const ctx = canvas.getContext('2d');
    ctx.fillStyle = '#f42020';
    ctx.beginPath();
    ctx.ellipse(canvas.width - markerSize, markerSize, markerSize, markerSize, 0, 0, 2 * Math.PI);
    ctx.closePath();
    ctx.fill();
    element.setAttribute(icon.unread, canvas.toDataURL());
}
async function discoverIcons(element) {
    if (element) {
        return createIcons(element, element.getAttribute('src'));
    }
    console.warn('Could not discover profile picture. Falling back to default image.');
    // Fall back to messenger favicon
    const messengerIcon = document.querySelector('link[rel~="icon"]');
    if (messengerIcon) {
        return createIcons(element, messengerIcon.getAttribute('href'));
    }
    // Fall back to facebook favicon
    return createIcons(element, 'https://facebook.com/favicon.ico');
}
async function getIcon(element, unread) {
    if (element === null) {
        return icon.read;
    }
    if (!element.getAttribute(icon.read)) {
        await discoverIcons(element);
    }
    return element.getAttribute(unread ? icon.unread : icon.read);
}
async function getLabel(element) {
    if ((0, lodash_1.isNull)(element)) {
        return '';
    }
    const emojis = [];
    if (element !== null) {
        for (const element_curr of element.children) {
            emojis.push(element_curr);
        }
    }
    for (const emoji of emojis) {
        emoji.outerHTML = emoji.querySelector('img')?.getAttribute('alt') ?? '';
    }
    return element.textContent ?? '';
}
async function createConversationNewDesign(element) {
    const conversation = {};
    // TODO: Exclude muted conversations
    /*
    const muted = Boolean(element.querySelector(selectors.muteIconNewDesign));
    */
    conversation.selected = Boolean(element.querySelector('[role=row] [role=link] > div:only-child'));
    conversation.unread = Boolean(element.querySelector('[aria-label="Mark as Read"]'));
    const unparsedLabel = element.querySelector('.a8c37x1j.ni8dbmo4.stjgntxs.l9j0dhe7 > span > span');
    conversation.label = await getLabel(unparsedLabel);
    const iconElement = element.querySelector('img');
    conversation.icon = await getIcon(iconElement, conversation.unread);
    return conversation;
}
async function createConversationList() {
    const conversationListSelector = selectors_1.default.conversationList;
    const list = await (0, element_ready_1.default)(conversationListSelector, {
        stopOnDomReady: false,
    });
    if (!list) {
        console.error('Could not find conversation list', conversationListSelector);
        return [];
    }
    const elements = [...list.children];
    // Remove last element from childer list
    elements.splice(-1, 1);
    const conversations = await Promise.all(elements.map(async (element) => createConversationNewDesign(element)));
    return conversations;
}
async function sendConversationList() {
    const conversationsToRender = await createConversationList();
    electron_better_ipc_1.ipcRenderer.callMain('conversations', conversationsToRender);
}
exports.sendConversationList = sendConversationList;
function genStringFromNode(element) {
    const cloneElement = element.cloneNode(true);
    let emojiString;
    const images = cloneElement.querySelectorAll('img');
    for (const image of images) {
        emojiString = image.alt;
        // Replace facebook's thumbs up with emoji
        if (emojiString === '(Y)' || emojiString === '(y)') {
            emojiString = '👍';
        }
        image.parentElement?.replaceWith(document.createTextNode(emojiString));
    }
    return cloneElement.textContent ?? undefined;
}
function countUnread(mutationsList) {
    const alreadyChecked = [];
    const unreadMutations = mutationsList.filter(mutation => 
    // When a conversations "becomes unread".
    (mutation.type === 'childList'
        && mutation.addedNodes.length > 0
        && (mutation.addedNodes[0].className === selectors_1.default.conversationSidebarUnreadDot))
        // When text is received
        || (mutation.type === 'characterData'
            // Make sure the text corresponds to a conversation
            && mutation.target.parentElement?.parentElement?.parentElement?.className === selectors_1.default.conversationSidebarTextParent)
        // When an emoji is received, node(s) are added
        || (mutation.type === 'childList'
            // Make sure the mutation corresponds to a conversation
            && mutation.target.parentElement?.parentElement?.className === selectors_1.default.conversationSidebarTextParent)
        // Emoji change
        || (mutation.type === 'attributes'
            && mutation.target.parentElement?.parentElement?.parentElement?.parentElement?.className === selectors_1.default.conversationSidebarTextParent));
    // Check latest mutation first
    for (const mutation of unreadMutations.reverse()) {
        const curr = mutation.target.parentElement.closest(selectors_1.default.conversationSidebarSelector);
        const href = curr.closest('[role="link"]')?.getAttribute('href');
        if (!href) {
            continue;
        }
        // It is possible to have multiple mutations for the same conversation, but we only want one notification.
        // So if the current conversation has already been checked, continue.
        // Additionally if the conversation is not unread, then also continue.
        if (alreadyChecked.includes(href) || !curr.querySelector('.' + selectors_1.default.conversationSidebarUnreadDot.replace(/ /g, '.'))) {
            continue;
        }
        alreadyChecked.push(href);
        // Get the image data URI from the parent of the author/text
        const imgUrl = curr.querySelector('img')?.dataset.caprineIcon;
        const textOptions = curr.querySelectorAll(selectors_1.default.conversationSidebarTextSelector);
        // Get the author and text of the new message
        const titleTextNode = textOptions[0];
        const bodyTextNode = textOptions[1];
        const titleText = genStringFromNode(titleTextNode);
        const bodyText = genStringFromNode(bodyTextNode);
        if (!bodyText || !titleText || !imgUrl) {
            continue;
        }
        // Send a notification
        electron_better_ipc_1.ipcRenderer.callMain('notification', {
            id: 0,
            title: titleText,
            body: bodyText,
            icon: imgUrl,
            silent: false,
        });
    }
}
async function updateTrayIcon() {
    const chatsIcon = await (0, element_ready_1.default)(selectors_1.default.chatsIcon, {
        stopOnDomReady: false,
    });
    // Extract messageCount from ariaLabel
    const messageCount = chatsIcon?.ariaLabel?.match(/\d+/g) ?? 0;
    electron_better_ipc_1.ipcRenderer.callMain('update-tray-icon', messageCount);
}
window.addEventListener('load', async () => {
    const sidebar = await (0, element_ready_1.default)('[role=navigation]:has([role=grid])', { stopOnDomReady: false });
    const leftSidebar = await (0, element_ready_1.default)(`${selectors_1.default.leftSidebar}:has(${selectors_1.default.chatsIcon})`, { stopOnDomReady: false });
    if (sidebar) {
        const conversationListObserver = new MutationObserver(async () => sendConversationList());
        const conversationCountObserver = new MutationObserver(countUnread);
        conversationListObserver.observe(sidebar, {
            subtree: true,
            childList: true,
            attributes: true,
            attributeFilter: ['class'],
        });
        conversationCountObserver.observe(sidebar, {
            characterData: true,
            subtree: true,
            childList: true,
            attributes: true,
            attributeFilter: ['src', 'alt'],
        });
    }
    if (leftSidebar) {
        const chatsIconObserver = new MutationObserver(async () => updateTrayIcon());
        chatsIconObserver.observe(leftSidebar, {
            subtree: true,
            childList: true,
            attributes: true,
            attributeFilter: ['aria-label'],
        });
    }
});
