/**********************************************************************
This software program is available to you under a choice of one of two 
licenses. You may choose to be licensed under either the GNU General Public 
License (GPL) Version 2, June 1991, available at 
http://www.fsf.org/copyleft/gpl.html, or the Intel BSD + Patent License, the 
text of which follows:

Recipient has requested a license and Intel Corporation ("Intel") is willing
to grant a license for the software entitled Linux Base Driver for the 
Intel(R) PRO/100 Family of Adapters (e100) (the "Software") being provided 
by Intel Corporation. The following definitions apply to this license:

"Licensed Patents" means patent claims licensable by Intel Corporation which 
are necessarily infringed by the use of sale of the Software alone or when 
combined with the operating system referred to below.

"Recipient" means the party to whom Intel delivers this Software.

"Licensee" means Recipient and those third parties that receive a license to 
any operating system available under the GNU Public License version 2.0 or 
later.

Copyright (c) 1999 - 2002 Intel Corporation.
All rights reserved.

The license is provided to Recipient and Recipient's Licensees under the 
following terms.

Redistribution and use in source and binary forms of the Software, with or 
without modification, are permitted provided that the following conditions 
are met:

Redistributions of source code of the Software may retain the above 
copyright notice, this list of conditions and the following disclaimer.

Redistributions in binary form of the Software may reproduce the above 
copyright notice, this list of conditions and the following disclaimer in 
the documentation and/or materials provided with the distribution.

Neither the name of Intel Corporation nor the names of its contributors 
shall be used to endorse or promote products derived from this Software 
without specific prior written permission.

Intel hereby grants Recipient and Licensees a non-exclusive, worldwide, 
royalty-free patent license under Licensed Patents to make, use, sell, offer 
to sell, import and otherwise transfer the Software, if any, in source code 
and object code form. This license shall include changes to the Software 
that are error corrections or other minor changes to the Software that do 
not add functionality or features when the Software is incorporated in any 
version of an operating system that has been distributed under the GNU 
General Public License 2.0 or later. This patent license shall apply to the 
combination of the Software and any operating system licensed under the GNU 
Public License version 2.0 or later if, at the time Intel provides the 
Software to Recipient, such addition of the Software to the then publicly 
available versions of such operating systems available under the GNU Public 
License version 2.0 or later (whether in gold, beta or alpha form) causes 
such combination to be covered by the Licensed Patents. The patent license 
shall not apply to any other combinations which include the Software. NO 
hardware per se is licensed hereunder.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
IMPLIED WARRANTIES OF MECHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL INTEL OR IT CONTRIBUTORS BE LIABLE FOR ANY 
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
ANY LOSS OF USE; DATA, OR PROFITS; OR BUSINESS INTERUPTION) HOWEVER CAUSED 
AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY OR 
TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**********************************************************************/

/**********************************************************************
*                                                                     *
* INTEL CORPORATION                                                   *
*                                                                     *
* This software is supplied under the terms of the license included   *
* above.  All use of this driver must be in accordance with the terms *
* of that license.                                                    *
*                                                                     *
* Module Name:  e100_config.c                                         *
*                                                                     *
* Abstract:     Functions for configuring the network adapter.        *
*                                                                     *
* Environment:  This file is intended to be specific to the Linux     *
*               operating system.                                     *
*                                                                     *
**********************************************************************/

#include "e100_config.h"

/* global variables */
static u8 e100_cfg_parm6 = CFG_BYTE_PARM6;

static u8 e100_rx_fifo_lmt =
(RX_FIFO_LMT < 0) ? DEFAULT_RX_FIFO_LIMIT : ((RX_FIFO_LMT >
                                              15) ? DEFAULT_RX_FIFO_LIMIT : RX_FIFO_LMT);

static u8 e100_tx_fifo_lmt =
(TX_FIFO_LMT < 0) ? DEFAULT_TX_FIFO_LIMIT : ((TX_FIFO_LMT >
                                              7) ? DEFAULT_TX_FIFO_LIMIT : TX_FIFO_LMT);

static u8 e100_rx_dma_cnt =
(RX_DMA_CNT < 0) ? CB_557_CFIG_DEFAULT_PARM4 : ((RX_DMA_CNT >
                                                 63) ? CB_557_CFIG_DEFAULT_PARM4 : RX_DMA_CNT);

static u8 e100_tx_dma_cnt =
(TX_DMA_CNT < 0) ? CB_557_CFIG_DEFAULT_PARM5 : ((TX_DMA_CNT >
                                                 63) ? CB_557_CFIG_DEFAULT_PARM5 : TX_DMA_CNT);

static u8 e100_urun_retry =
(TX_UR_RETRY < 0) ? CB_557_CFIG_DEFAULT_PARM5 : ((TX_UR_RETRY >
                                                  3) ? CB_557_CFIG_DEFAULT_PARM5 : TX_UR_RETRY);

/* e100_MWI_enable
 * 
 * Enable/Disable use of Memory Write and Invalidate
 * 0 - Disable
 * 1 - Enable
 *
 * Note: This should be enabled only for an 82558 based adapter (PRO/100+)
 * and only on systems in which the PCI bus supports MWI. If enabled on a 
 * system that does not support MWI, performance might be affected.
 */       
static int e100_MWI_enable = 1;

/* 
 * e100_read_align_enable
 * 
 * Enable/Disable use of Cache Line Read/Write Termination
 * 0 - Disable
 * 1 - Enable
 *
 * Note: This should be enabled only for an 82558 based adapter (PRO/100+)
 * and only on cache line oriented systems. If enabled on a
 * system that is not cache line oriented, performance might be affected.
 */
static int e100_read_align_enable = 0;


/* extern variables */
extern int e100_speed_duplex[MAX_NIC];
extern int flow_control[MAX_NIC];
extern int XsumRX[MAX_NIC];

/*
 * Procedure:   e100_config_init
 *
 * Description: This routine will initialize the card's configure block.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *
 * Returns:
 */
void e100_config_init(bd_config_t *bdp)
{
    /* initialize config block */
    memset(bdp->config, 0x00, sizeof(bdp->config));

    /* First fill in the static (end user can't change) config bytes */
    bdp->config[0] = CB_CFIG_BYTE_COUNT;
    bdp->config[2] = CB_557_CFIG_DEFAULT_PARM2;
    bdp->config[3] = CB_557_CFIG_DEFAULT_PARM3;
    /* Commented out byte 6 for CNA intr fix. */
    /* bdp->config[6]  = CB_557_CFIG_DEFAULT_PARM6; */
    bdp->config[9] = CB_557_CFIG_DEFAULT_PARM9;
    bdp->config[10] = CB_557_CFIG_DEFAULT_PARM10;
    bdp->config[11] = CB_557_CFIG_DEFAULT_PARM11;
    bdp->config[12] = CB_557_CFIG_DEFAULT_PARM12;
    bdp->config[13] = CB_557_CFIG_DEFAULT_PARM13;
    bdp->config[14] = CB_557_CFIG_DEFAULT_PARM14;
    bdp->config[18] = CB_557_CFIG_DEFAULT_PARM18;
    bdp->config[20] = CB_557_CFIG_DEFAULT_PARM20;
    bdp->config[21] = CB_557_CFIG_DEFAULT_PARM21;
    
    /* Now fill in the rest of the configuration bytes (the bytes that *
     * contain user configurable parameters). */

    /* Change for 82558 enhancement */
    /* Set the Tx and Rx Fifo limits */
    if ((bdp->flags & IS_BACHELOR) && (e100_tx_fifo_lmt < 8)) {
        e100_tx_fifo_lmt = 8;   /* set 8 as the minimum */
    }
    
    bdp->config[1] = 
        (u8) (BIT_7 | (e100_tx_fifo_lmt << 4) | e100_rx_fifo_lmt);
    
    e100_config_ifs(bdp);
    
    /* Change for 82558 enhancement */
    /* MWI enable. This should be turned on only if enabled in * e100.c and
     * if the adapter is a 82558/9 and if the PCI command reg. * has enabled
     * the MWI bit. */
    if ((bdp->flags & IS_BACHELOR) && (e100_MWI_enable)) {
        bdp->config[3] |= CB_CFIG_MWI_EN;
    }
    
    /* Read Align/Write Terminate on cache line. This should be * turned on
     * only if enabled in e100.c and if the adapter is a 82558/9 * and if the 
     * system is cache line oriented. */
    if ((bdp->flags & IS_BACHELOR) && (e100_read_align_enable))
        bdp->config[3] |= CB_CFIG_READAL_EN | CB_CFIG_TERMCL_EN;
    
    /* Set the Tx and Rx DMA maximum byte count fields. */
    bdp->config[4] = e100_rx_dma_cnt;
    bdp->config[5] = e100_tx_dma_cnt;
    if ((e100_rx_dma_cnt) || (e100_tx_dma_cnt)) {
        bdp->config[5] |= CB_CFIG_DMBC_EN;
    }
    
    /* Change for 82558 enhancement */
    /* Extended TCB. Should be turned on only if enabled * in e100.c and if
     * the adapter is a 82558/9. */
    if ((e100_cfg_parm6 == 0x32) || (e100_cfg_parm6 == 0x3a)) {
        bdp->config[6] = e100_cfg_parm6;
    } else {
        bdp->config[6] = CB_557_CFIG_DEFAULT_PARM6;
    }
    
    if (bdp->flags & IS_BACHELOR) {
        bdp->config[6] &= ~CB_CFIG_EXT_TCB_DIS;
    }
    
    /*
     * Enable extended statistical counters (82558 and up) and TCO counters
     * (82559 and up) and set the statistical counters' mode in bdp 
     *  
     *  stat. mode      |    TCO stat. bit (2)  |  Extended stat. bit (5)
     * ------------------------------------------------------------------
     *  Basic (557)     |       0               |         1
     * ------------------------------------------------------------------
     *  Extended (558)  |       0               |         0
     * ------------------------------------------------------------------
     *  TCO (559)       |       1               |         1
     * ------------------------------------------------------------------
     *  Reserved        |       1               |         0
     * ------------------------------------------------------------------
     */
    if (bdp->flags & IS_BACHELOR) {  /* This is 558 and up - enable extended counters */
        
        if (bdp->rev_id >= D101MA_REV_ID) {   /* this is 559 and up - enable TCO counters */
            bdp->config[6] |= CB_CFIG_TCO_STAT;
            bdp->config[6] |= CB_CFIG_EXT_STAT_DIS;
            bdp->stat_mode = E100_TCO_STATS;
        
        } else {  /* this is 558 */
            bdp->config[6] &= ~CB_CFIG_TCO_STAT;
            bdp->config[6] &= ~CB_CFIG_EXT_STAT_DIS;
            bdp->stat_mode = E100_EXTENDED_STATS;
        }
        
    } else {  /* this is 557 */
        bdp->config[6] &= ~CB_CFIG_TCO_STAT;
       	bdp->config[6] |= CB_CFIG_EXT_STAT_DIS;
        bdp->stat_mode = E100_BASIC_STATS;
    }
    
    /* Set up number of retries after under run */
    bdp->config[7] = ((CB_557_CFIG_DEFAULT_PARM7 & (~CB_CFIG_URUN_RETRY)) |
                      (e100_urun_retry << 1));
    
    /* Change for 82558 enhancement */
    /* Dynamic TBD. Should be turned on only if enabled in e100.c and if
     * the adapter is a 82558/9. */
    if (bdp->flags & IS_BACHELOR)
        bdp->config[7] |= CB_CFIG_DYNTBD_EN;
    
    /* Setup for MII or 503 operation.  The CRS+CDT bit should only be set */
    /* when operating in 503 mode. */
    if (bdp->phy_addr == 32) {
        bdp->config[8] = (CB_557_CFIG_DEFAULT_PARM8 & (~CB_CFIG_503_MII));
        bdp->config[15] = (CB_557_CFIG_DEFAULT_PARM15 | CB_CFIG_CRS_OR_CDT);
    } else {
        bdp->config[8] = (CB_557_CFIG_DEFAULT_PARM8 | CB_CFIG_503_MII);
        bdp->config[15] = (CB_557_CFIG_DEFAULT_PARM15 & (~CB_CFIG_CRS_OR_CDT));
    }
    
    /* Change for 82558 enhancement */
    /* enable flow control only if not 557 */
    if ((bdp->flags & IS_BACHELOR) && (flow_control[bdp->bd_number] == TRUE)) {
        bdp->config[16] = DFLT_FC_DELAY_LSB;
        bdp->config[17] = DFLT_FC_DELAY_MSB;
        /* Removed CB_CFIG_TX_FC_EN frm line below. This bit has to be 0 to 
         * enable flow control. */
        bdp->config[19] = (CB_557_CFIG_DEFAULT_PARM19 | CB_CFIG_FC_RESTOP |
                           CB_CFIG_FC_RESTART | CB_CFIG_REJECT_FC);
    } else {
        bdp->config[16] = CB_557_CFIG_DEFAULT_PARM16;
        bdp->config[17] = CB_557_CFIG_DEFAULT_PARM17;
        bdp->config[19] = CB_557_CFIG_DEFAULT_PARM19 | CB_CFIG_TX_FC_DIS;
    }
    
    e100_config_force_dplx(bdp);
    e100_config_promisc(bdp, B_FALSE);
    e100_config_mulcast_enbl(bdp, B_FALSE);
    
    /* Enable checksum offloading if we are on a supported adapter. */
    if ((bdp->rev_id >= D101MA_REV_ID) && (bdp->rev_id < D102_REV_ID) &&
        (XsumRX[bdp->bd_number] == TRUE) && (bdp->ppci_dev->device != 0x1209)) {
        bdp->flags |= DF_CSUM_OFFLOAD;
        bdp->config[9] |= 1;
    } else if (bdp->rev_id >= D102_REV_ID) {
        /* The D102 chip allows for 32 config bytes.  This value is
           supposed to be in Byte 0.  Just add the extra bytes to
           what was already setup in the block. */
        bdp->config[0] += CB_CFIG_D102_BYTE_COUNT;
        
        /* now we need to enable the extended RFD.  When this is
           enabled, the immediated receive data buffer starts at offset
           32 from the RFD base address, instead of at offset 16. */
        bdp->config[7] |= CB_CFIG_EXTENDED_RFD;
        
        /* put the chip into D102 receive mode.  This is neccessary
           for any parsing and offloading features. */
        bdp->config[22] = CB_CFIG_RECEIVE_GAMLA_MODE;
        
        /* set the flag if checksum offloading was enabled */
        if (XsumRX[bdp->bd_number] == TRUE) {
            bdp->flags |= DF_CSUM_OFFLOAD;
        }
    }
}

/*
 * Procedure:   e100_force_config
 *
 * Description: This routine will force a configure command to the 82557.
 *              This command will be executed in polled mode as interrupts
 *              are _disabled_ at this time.  The configuration parameters
 *              that are user configurable will have been set in "e100.c".
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *
 * Returns:
 *      B_TRUE - If the configure command was successfully issued and completed
 *      B_FALSE - If the configure command failed to complete properly
 */
unsigned char
e100_force_config(bd_config_t *bdp)
{
    spin_lock_bh(&(bdp->config_lock));
    
    bdp->config[0] = CB_CFIG_BYTE_COUNT;
    if (bdp->rev_id >= D102_REV_ID) {
        /* The D102 chip allows for 32 config bytes.  This value is
           supposed to be in Byte 0.  Just add the extra bytes to
           what was already setup in the block. */
        bdp->config[0] += CB_CFIG_D102_BYTE_COUNT;
    }

    spin_unlock_bh(&(bdp->config_lock));
    // although we call config outside the lock, there is no
    // race condition because config byte count has maximum value
    return e100_config(bdp);
}

/*
 * Procedure:   e100_config
 *
 * Description: This routine will issue a configure command to the 82557.
 *              This command will be executed in polled mode as interrupts
 *              are _disabled_ at this time.  The configuration parameters
 *              that are user configurable will have been set in "e100.c".
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *
 * Returns:
 *      B_TRUE - If the configure command was successfully issued and completed
 *      B_FALSE - If the configure command failed to complete properly
 */
unsigned char
e100_config(bd_config_t *bdp)
{
        pcb_header_t    pntcb_hdr;
        unsigned char	res;

        spin_lock_bh(&bdp->bd_tx_lock);

        pntcb_hdr = (pcb_header_t) bdp->pntcb;   /* get hdr of non tcb cmd */

        pntcb_hdr->cb_cmd = __constant_cpu_to_le16(CB_CONFIGURE);

        spin_lock_bh(&bdp->config_lock);
        
        if (bdp->config[0] == 0) {
                spin_unlock_bh(&bdp->config_lock);
                spin_unlock_bh(&bdp->bd_tx_lock);
                e100_tx_notify_start(bdp, SHORT_STOP);
                return B_TRUE;
        }

        if (bdp->config[0] < CB_CFIG_MIN_PARAMS) {
            bdp->config[0] = CB_CFIG_MIN_PARAMS;
        }
        
        /* Copy the device's config block to the device's memory */
        memcpy(bdp->pntcb->ntcb.config.cfg_byte, bdp->config, bdp->config[0]);

        if ( (res = e100_exec_non_cu_cmd(bdp)) == B_TRUE) {
                /* reset number of bytes to config next time */
                bdp->config[0] = 0;
        }

        spin_unlock_bh(&bdp->config_lock);
        spin_unlock_bh(&bdp->bd_tx_lock);
        e100_tx_notify_start(bdp, SHORT_STOP);
        return res;
}

/*
 * Procedure:   e100_config_promisc
 *
 * Description: This routine will enable or disable promiscuous mode
 *              in the adapter's congif block, as indicated by the
 *              adapter's e100_bdconfig structure.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *      enable - should we enable this option or not
 *
 * Returns:
 */
void
e100_config_promisc(bd_config_t *bdp, unsigned char enable)
{
    spin_lock_bh(&(bdp->config_lock));
    
    /* if in promiscuous mode, save bad frames */
    if (enable == B_TRUE) {
        
        if (!(bdp->config[6] & CB_CFIG_SAVE_BAD_FRAMES)) {
            bdp->config[6] |= CB_CFIG_SAVE_BAD_FRAMES;
            E100_CONFIG(bdp, 6);
        }
        
        if (bdp->config[7] & (u8)BIT_0) {
            bdp->config[7] &= (u8) (~BIT_0);
            E100_CONFIG(bdp, 7);
        }
        
        if (!(bdp->config[15] & CB_CFIG_PROMISCUOUS)) {
            bdp->config[15] |= CB_CFIG_PROMISCUOUS;
            E100_CONFIG(bdp, 15);
        }
    
    } else { /* not in promiscuous mode */
        
        if (bdp->config[6] & CB_CFIG_SAVE_BAD_FRAMES) {
            bdp->config[6] &= ~CB_CFIG_SAVE_BAD_FRAMES;
            E100_CONFIG(bdp, 6);
        }
        
        if (!(bdp->config[7] & (u8)BIT_0)) {
            bdp->config[7] |= (u8) (BIT_0);
            E100_CONFIG(bdp, 7);
        }
        
        if (bdp->config[15] & CB_CFIG_PROMISCUOUS) {
            bdp->config[15] &= ~CB_CFIG_PROMISCUOUS;
            E100_CONFIG(bdp, 15);
        }
    }
    
    spin_unlock_bh(&(bdp->config_lock));
}


/*
 * Procedure:   e100_config_mulcast_enbl
 *
 * Description: This routine will enable or disable multicast packets
 *              in the adapter's congif block, as indicated by the
 *              adapter's e100_bdconfig structure.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *      enable - should we enable this option or not
 *
 * Returns:
 */
void
e100_config_mulcast_enbl(bd_config_t *bdp, unsigned char enable)
{
    spin_lock_bh(&(bdp->config_lock));
    
    /* this flag is used to enable receiving all multicast packet */
    if (enable == B_TRUE) {
        if (!(bdp->config[21] & CB_CFIG_MULTICAST_ALL)) {
            bdp->config[21] |= CB_CFIG_MULTICAST_ALL;
            E100_CONFIG(bdp, 21);
        }
        
    } else {
        if (bdp->config[21] & CB_CFIG_MULTICAST_ALL) {
            bdp->config[21] &= ~CB_CFIG_MULTICAST_ALL;
            E100_CONFIG(bdp, 21);
        }
    }
    
    spin_unlock_bh(&(bdp->config_lock));
}

/*
 * Procedure:   e100_config_ifs
 *
 * Description: This routine will configure the adaptive IFS value
 *              in the adapter's congif block, as indicated by the
 *              adapter's e100_bdconfig structure.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *
 * Returns:
 */
void
e100_config_ifs(bd_config_t *bdp)
{
    u8         value = 0;
    
    spin_lock_bh(&(bdp->config_lock));
    
    /* IFS value is only needed to be specified at half-duplex mode */
    if (bdp->cur_dplx_mode == HALF_DUPLEX) {
        value = (u8) bdp->ifs_value;
    }
    
    if (bdp->config[2] != value) {
        bdp->config[2] = value;
        E100_CONFIG(bdp, 2);
    }
    
    spin_unlock_bh(&(bdp->config_lock));
}

/*
 * Procedure:   e100_config_force_dplx
 *
 * Description: This routine will enable or disable force full duplex
 *              in the adapter's congif block, as indicated by the
 *              adapter's e100_bdconfig structure.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *
 * Returns:
 */
void
e100_config_force_dplx(bd_config_t *bdp)
{
    
    spin_lock_bh(&(bdp->config_lock));
    
    /* We must force full duplex on if we are using PHY 0, and we are */
    /* supposed to run in FDX mode.  We do this because the e100 has only */
    /* one FDX# input pin, and that pin will be connected to PHY 1. */
    /* Changed the 'if' condition below to fix performance problem * at 10
     * full. The Phy was getting forced to full duplex while the MAC * was
     * not, because the cur_dplx_mode was not being set to 2 by SetupPhy. *
     * This is how the condition was, initially. * This has been changed so
     * that the MAC gets forced to full duplex * simply if the user has
     * forced full duplex. * * if (( bdp->phy_addr == 0 ) && (
     * bdp->cur_dplx_mode == 2 )) */
    /* The rest of the fix is in the PhyDetect code.          */
    if ((e100_speed_duplex[bdp->bd_number] == 2) ||
        (e100_speed_duplex[bdp->bd_number] == 4) ||
        ((bdp->phy_addr == 32) && (bdp->cur_dplx_mode == FULL_DUPLEX))) {
        if (!(bdp->config[19] & CB_CFIG_FORCE_FDX)) {
            bdp->config[19] |= CB_CFIG_FORCE_FDX;
            E100_CONFIG(bdp, 19);
        }
        
    } else {
        if (bdp->config[19] & CB_CFIG_FORCE_FDX) {
            bdp->config[19] &= ~CB_CFIG_FORCE_FDX;
            E100_CONFIG(bdp, 19);
        }
    }
    
    spin_unlock_bh(&(bdp->config_lock));
}

#ifdef IANS_BASE_VLAN_TAGGING

/*
 * Procedure:   e100_config_long_rx
 *
 * Description: This routine will enable or disable reception of larger packets.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *      enable - should we enable this option or not
 *
 * Returns:
 */
static void
e100_config_long_rx(bd_config_t *bdp, unsigned char enable)
{
    if (enable == B_TRUE) {
        if (!(bdp->config[18] & CB_CFIG_LONG_RX_OK)) {
            bdp->config[18] |= CB_CFIG_LONG_RX_OK;
            E100_CONFIG(bdp, 18);
        }
        
    } else {
        if ((bdp->config[18] & CB_CFIG_LONG_RX_OK)) {
            bdp->config[18] &= ~CB_CFIG_LONG_RX_OK;
            E100_CONFIG(bdp, 18);
        }
    }
}

/*
 * Procedure:   e100_config_vlan_drop
 *
 * Description: This routine will enable or disable vlan tag stripping.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *      enable - should we enable this option or not
 *
 * Returns:
 */
static void
e100_config_vlan_drop(bd_config_t *bdp, unsigned char enable)
{
    if (enable == B_TRUE) {
        if (!(bdp->config[22] & CB_CFIG_VLAN_DROP_ENABLE)) {
            bdp->config[22] |= CB_CFIG_VLAN_DROP_ENABLE;
            E100_CONFIG(bdp, 22);
        }
        
    } else {
        if ((bdp->config[22] & CB_CFIG_VLAN_DROP_ENABLE)) {
            bdp->config[22] &= ~CB_CFIG_VLAN_DROP_ENABLE;
            E100_CONFIG(bdp, 22);
        }
    }
}

/*
 * Procedure:   e100_config_enable_tagging
 *
 * Description: This routine will enable or disable vlan tagging.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *      enable - should we enable this option or not
 *
 * Returns:
 */
void
e100_config_enable_tagging(bd_config_t *bdp, unsigned char enable)
{
    spin_lock_bh(&(bdp->config_lock));
    
    if (bdp->rev_id >= D101B0_REV_ID)
        e100_config_long_rx(bdp, enable);
    
    if (bdp->rev_id >= D102_REV_ID)
        e100_config_vlan_drop(bdp, enable);
    
    spin_unlock_bh(&(bdp->config_lock));
}
#endif /* IANS_BASE_VLAN_TAGGING */

#ifdef IDIAG_PRO_SUPPORT

static unsigned char
e100_config_loopback_change_block(bd_config_t *bdp,u32 mode)
{
        if( (bdp->config[10] & CB_CFIG_LOOPBACK_MODE)!= 
            mode){

                bdp->config[10] &= (~CB_CFIG_LOOPBACK_MODE);
                bdp->config[10] |=mode;
               	E100_CONFIG(bdp, 10);
            	return B_TRUE;

        }
        return B_FALSE;
}

/*
 * Procedure:   e100_config_loopback_mode
 *
 * Description: This routine will change the loopback mode
 *              in the adapter's congif block, as indicated by the
 *              loopback mode argument.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *		mode   - loopback mode(internal/external/normal)
 *
 * Returns: True -if the config block was changed
 */

unsigned char
e100_config_loopback_mode(bd_config_t *bdp,u32 mode)
{
    unsigned char	bc_changed = B_FALSE;

    spin_lock_bh(&(bdp->config_lock));
    
    switch(mode){
        
    case(IDIAG_E100_DIAG_MAC_LB):
        bc_changed = e100_config_loopback_change_block(bdp,CB_CFIG_LOOPBACK_INTERNAL);
        break;
        
    case(IDIAG_E100_DIAG_EXT_LB):
        bc_changed = e100_config_loopback_change_block(bdp,CB_CFIG_LOOPBACK_EXTERNAL);
        break;
    
    case(IDIAG_E100_DIAG_TCVR_LB):
        bc_changed = e100_config_loopback_change_block(bdp,CB_CFIG_LOOPBACK_NORMAL);
        break;
        
    default:
        printk(KERN_NOTICE "e100_config_loopback_mode:Invalid argument 'mode': %d\n",mode);
    }
    
    spin_unlock_bh(&(bdp->config_lock));
    return bc_changed;
}


/*
 * Procedure:   e100_config_tcb_ext_enable
 *
 * Description: This routine will enable/disable the tcb extended mode
 *              in the adapter's congif block, as indicated by the
 *              tcb mode argument.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *		enable   - tcb extended mode enable
 *
 * Returns: True -if the config block was changed
 */

unsigned char
e100_config_tcb_ext_enable(bd_config_t *bdp, unsigned char enable)
{
    unsigned char	bc_changed = B_FALSE;

    spin_lock_bh(&(bdp->config_lock));

    if (enable) {
        if( bdp->config[6] & CB_CFIG_EXT_TCB_DIS) {
            
            bdp->config[6] &= (~CB_CFIG_EXT_TCB_DIS);
            E100_CONFIG(bdp, 6);
            bc_changed =B_TRUE;
       	} 

    } else {
        if( !(bdp->config[6] & CB_CFIG_EXT_TCB_DIS)) {
            
            bdp->config[6] |= CB_CFIG_EXT_TCB_DIS;
            E100_CONFIG(bdp, 6);
            bc_changed = B_TRUE;
       	}
    }
    spin_unlock_bh(&(bdp->config_lock));
    
    return bc_changed;
}

/*
 * Procedure:   e100_config_dynamic_tbd
 *
 * Description: This routine will enable/disable the dinamic tbd mode
 *              in the adapter's congif block, as indicated by the
 *             eboolean enable argument.
 *
 * Arguments:
 *      bdp    - Ptr to this card's e100_bdconfig structure
 *		enable   - True to enable
 *
 * Returns: True -if the config block was changed
 */
unsigned char
e100_config_dynamic_tbd(bd_config_t *bdp,unsigned char enable)
{
    unsigned char	bc_changed = B_FALSE;
    
    spin_lock_bh(&(bdp->config_lock));
    
    if (enable) {
        if( !(bdp->config[7] & CB_CFIG_DYNTBD_EN)) {
            
            bdp->config[7] |= CB_CFIG_DYNTBD_EN;
            E100_CONFIG(bdp, 7);
            bc_changed =B_TRUE;
       	} 

    } else {
    	if( bdp->config[7] & CB_CFIG_DYNTBD_EN) {
            
            bdp->config[7] &= (~CB_CFIG_DYNTBD_EN);
            E100_CONFIG(bdp, 7);
            bc_changed =B_TRUE;
       	} 
    }
    spin_unlock_bh(&(bdp->config_lock));

    return bc_changed;
}

#endif /* IDIAG_PRO_SUPPORT */
