/**********************************************************************
This software program is available to you under a choice of one of two 
licenses. You may choose to be licensed under either the GNU General Public 
License (GPL) Version 2, June 1991, available at 
http://www.fsf.org/copyleft/gpl.html, or the Intel BSD + Patent License, the 
text of which follows:

Recipient has requested a license and Intel Corporation ("Intel") is willing
to grant a license for the software entitled Linux Base Driver for the 
Intel(R) PRO/100 Family of Adapters (e100) (the "Software") being provided 
by Intel Corporation. The following definitions apply to this license:

"Licensed Patents" means patent claims licensable by Intel Corporation which 
are necessarily infringed by the use of sale of the Software alone or when 
combined with the operating system referred to below.

"Recipient" means the party to whom Intel delivers this Software.

"Licensee" means Recipient and those third parties that receive a license to 
any operating system available under the GNU Public License version 2.0 or 
later.

Copyright (c) 1999 - 2002 Intel Corporation.
All rights reserved.

The license is provided to Recipient and Recipient's Licensees under the 
following terms.

Redistribution and use in source and binary forms of the Software, with or 
without modification, are permitted provided that the following conditions 
are met:

Redistributions of source code of the Software may retain the above 
copyright notice, this list of conditions and the following disclaimer.

Redistributions in binary form of the Software may reproduce the above 
copyright notice, this list of conditions and the following disclaimer in 
the documentation and/or materials provided with the distribution.

Neither the name of Intel Corporation nor the names of its contributors 
shall be used to endorse or promote products derived from this Software 
without specific prior written permission.

Intel hereby grants Recipient and Licensees a non-exclusive, worldwide, 
royalty-free patent license under Licensed Patents to make, use, sell, offer 
to sell, import and otherwise transfer the Software, if any, in source code 
and object code form. This license shall include changes to the Software 
that are error corrections or other minor changes to the Software that do 
not add functionality or features when the Software is incorporated in any 
version of an operating system that has been distributed under the GNU 
General Public License 2.0 or later. This patent license shall apply to the 
combination of the Software and any operating system licensed under the GNU 
Public License version 2.0 or later if, at the time Intel provides the 
Software to Recipient, such addition of the Software to the then publicly 
available versions of such operating systems available under the GNU Public 
License version 2.0 or later (whether in gold, beta or alpha form) causes 
such combination to be covered by the Licensed Patents. The patent license 
shall not apply to any other combinations which include the Software. NO 
hardware per se is licensed hereunder.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
IMPLIED WARRANTIES OF MECHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL INTEL OR IT CONTRIBUTORS BE LIABLE FOR ANY 
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
ANY LOSS OF USE; DATA, OR PROFITS; OR BUSINESS INTERUPTION) HOWEVER CAUSED 
AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY OR 
TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**********************************************************************/

/**********************************************************************
*                                                                       *
* INTEL CORPORATION                                                     *
*                                                                       *
* This software is supplied under the terms of the license included     *
* above.  All use of this driver must be in accordance with the terms   *
* of that license.                                                      *
*                                                                       *
* Module Name:  e100_proc.c                                             *
*                                                                       *
* Abstract:     Functions to handle the proc file system.               *
*               Create the proc directories and files and run read and  *
*               write requests from the user                            *
*                                                                       *
* Environment:  This file is intended to be specific to the Linux       *
*               operating system.                                       *
*                                                                       *
**********************************************************************/


#include "e100_proc.h"

/***************************************************************************/
/*       /proc File System Interaface Support Functions                    */
/***************************************************************************/


#ifdef CONFIG_PROC_FS
static struct proc_dir_entry *adapters_proc_dir = 0;

/* externs from e100_main.c */
extern const char *e100_short_driver_name;
extern const char *e100_version;
extern struct net_device_stats *e100_get_stats(struct net_device * dev);
extern char *e100_GetBrandingMesg(bd_config_t * bdp);
extern void e100_MdiWrite (bd_config_t *, u32, u32, u16);

static void e100_proc_cleanup(void);
static unsigned char e100_init_proc_dir(void);

#define E100_EOU

#ifdef E100_EOU
extern int e100_speed_duplex[MAX_NIC];
extern int RxDescriptors[MAX_NIC];
extern int TxDescriptors[MAX_NIC]; 
extern int XsumRX[MAX_NIC];
extern int ucode[MAX_NIC];
extern int IntDelay[MAX_NIC];
extern int BundleMax[MAX_NIC];
extern int BundleSmallFr[MAX_NIC];
extern int ber[MAX_NIC];
extern int flow_control[MAX_NIC];
extern int IFS[MAX_NIC];
#endif

#define ADAPTERS_PROC_DIR       "PRO_LAN_Adapters"
#define WRITE_BUF_MAX_LEN 20  
#define READ_BUF_MAX_LEN  256
#define E100_PE_LEN       25

/* flags */
#define PROC_LF 0x0002

typedef struct _e100_proc_entry
{
    char          name[256];
    read_proc_t  *read_proc;
    write_proc_t *write_proc;
    u16           flags;
} e100_proc_entry;


static int generic_read(char *page, char **start, off_t off, int count, int *eof)
{
    int len;

    len = strlen(page);
    page[len++] = '\n';

    if (len <= off + count)
        *eof = 1;
    *start = page + off;
    len -= off;
    if (len > count)
        len = count;
    if (len < 0)
        len = 0;
    return len;
}

static int read_ulong(char *page, char **start, off_t off, 
    int count, int *eof, unsigned long l)
{
    sprintf(page, "%lu", l);

    return generic_read(page, start, off, count, eof);
}

static int read_ulong_hex(char *page, char **start, off_t off, 
    int count, int *eof, unsigned long l)
{
    sprintf(page, "0x%04lx", l);

    return generic_read(page, start, off, count, eof);
}

static int read_generic_0(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    return read_ulong(page, start, off, count, eof, 
        (unsigned long)0); 
}

static int read_hwaddr(char *page, char **start, off_t off,
    int count, int *eof, unsigned char *hwaddr)
{
    sprintf(page, "%02X:%02X:%02X:%02X:%02X:%02X",
        hwaddr[0], hwaddr[1], hwaddr[2], 
        hwaddr[3], hwaddr[4], hwaddr[5]);

    return generic_read(page, start, off, count, eof);
}


static int read_descr(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
    char *msg;
        
    msg=e100_GetBrandingMesg(bdp);
    strncpy(page, msg, READ_BUF_MAX_LEN);

    return generic_read(page, start, off, count, eof);
}

static int read_drvr_name(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    strncpy(page, e100_short_driver_name, READ_BUF_MAX_LEN);

    return generic_read(page, start, off, count, eof);
}

static int read_drvr_ver(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    strncpy(page, e100_version, READ_BUF_MAX_LEN);

    return generic_read(page, start, off, count, eof);
}

static int read_pci_vendor(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong_hex(page, start, off, count, eof, 
        (unsigned long) bdp->ppci_dev->vendor);
}
        
static int read_pci_device(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong_hex(page, start, off, count, eof, 
        (unsigned long) bdp->ppci_dev->device);
}
        
static int read_pci_sub_vendor(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong_hex(page, start, off, count, eof, 
        (unsigned long) bdp->sub_ven_id);
}
        
static int read_pci_sub_device(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong_hex(page, start, off, count, eof, 
        (unsigned long) bdp->sub_dev_id);
}
        
static int read_pci_revision(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong_hex(page, start, off, count, eof, 
        (unsigned long) bdp->rev_id);
}

static int read_pci_bus(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof,
        (unsigned long) (bdp->ppci_dev->bus->number));
}

static int read_pci_slot(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof,
        (unsigned long)(PCI_SLOT((bdp->ppci_dev->devfn))));
}

static int read_irq(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
        (unsigned long)(bdp->device->irq));
}

static int read_dev_name(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    strncpy(page, bdp->device->name, READ_BUF_MAX_LEN);

    return generic_read(page, start, off, count, eof);
}

static int read_current_hwaddr(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
    unsigned char *hwaddr = bdp->device->dev_addr;

    return read_hwaddr(page, start, off, count, eof, hwaddr);
}

static int read_permanent_hwaddr(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
    unsigned char *hwaddr = bdp->perm_node_address;

    return read_hwaddr(page, start, off, count, eof, hwaddr);
}


static int read_part_number(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
    sprintf(page, "%06lx-%03x",
            (unsigned long)(bdp->pwa_no >>8),
            (unsigned int)(bdp->pwa_no & 0xFF));
    return generic_read(page, start, off, count, eof);
}

static int read_link_status(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    if (bdp->flags & DF_LINK_UP)
        sprintf(page, "up");
    else
        sprintf(page, "down");

    return generic_read(page, start, off, count, eof);
}

static int read_speed(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    if (bdp->flags & DF_LINK_UP)
        return read_ulong(page, start, off, count, eof,
                          (unsigned long) (bdp->cur_line_speed));

    sprintf(page, "N/A");
    return generic_read(page, start, off, count, eof);
}

static int read_dplx_mode(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
    char *dplx_mode;

    if (bdp->flags & DF_LINK_UP) {
        dplx_mode = (bdp->cur_dplx_mode == FULL_DUPLEX) ? "full" : "half";
    } else {
        dplx_mode = "N/A";
    }

    sprintf(page, "%s", dplx_mode);

    return generic_read(page, start, off, count, eof);
}

static int read_state(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    if (bdp->device->flags & IFF_UP)
        sprintf(page, "up");
    else
        sprintf(page, "down");

    return generic_read(page, start, off, count, eof);
}

static int read_rx_packets(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.rx_packets);
}

static int read_tx_packets(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.tx_packets);
}

static int read_rx_bytes(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.rx_bytes);
}

static int read_tx_bytes(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.tx_bytes);
}

static int read_rx_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;
        
    e100_get_stats(bdp->device);
    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.rx_errors);
}

static int read_tx_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    e100_get_stats(bdp->device);
    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.tx_errors);
}

static int read_rx_dropped(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.rx_dropped);
}

static int read_tx_dropped(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.tx_dropped);
}

static int read_rx_multicast_packets(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    sprintf(page, "N/A");
    return generic_read(page, start, off, count, eof);
}

static int read_collisions(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.collisions);
}

static int read_rx_length_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.rx_length_errors);
}

static int read_rx_over_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.rx_over_errors);
}

static int read_rx_crc_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.rx_crc_errors);
}

static int read_rx_frame_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.rx_frame_errors);
}

static int read_rx_fifo_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.rx_fifo_errors);
}

static int read_rx_missed_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.rx_missed_errors);
}

static int read_tx_aborted_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.tx_aborted_errors);
}

static int read_tx_carrier_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.tx_carrier_errors);
}

static int read_tx_fifo_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.tx_fifo_errors);
}

static int read_tx_abort_late_coll(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.tx_late_col);
}

static int read_tx_deferred_ok(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.tx_ok_defrd);
}

static int read_tx_single_coll_ok(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.tx_one_retry);
}

static int read_tx_multi_coll_ok(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.tx_mt_one_retry);
}

static int read_rx_align_errors(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.net_stats.rx_frame_errors);
}

static int read_tx_fc_packets (char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.xmt_fc_pkts);
}

static int read_rx_fc_packets (char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.rcv_fc_pkts);
}

static int read_rx_fc_unsupported(char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.rcv_fc_unsupported);
}

static int read_tx_tco_packets (char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.xmt_tco_pkts);
}

static int read_rx_tco_packets (char *page, char **start, off_t off,
    int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long) bdp->drv_stats.rcv_tco_pkts);
}

static void set_led(bd_config_t * bdp, u16 led_mdi_op)
{

  spin_lock_bh(&bdp->mdi_access_lock);
  
  e100_MdiWrite(bdp,PHY_82555_LED_SWITCH_CONTROL,
		bdp->phy_addr, led_mdi_op);
            
  spin_unlock_bh(&bdp->mdi_access_lock);

  set_current_state(TASK_INTERRUPTIBLE);
  schedule_timeout(E100_MAX(HZ/20,1)); // at least 50 ms

  spin_lock_bh(&bdp->mdi_access_lock);

  /* turn led ownership to the chip */
  e100_MdiWrite(bdp,PHY_82555_LED_SWITCH_CONTROL,
		bdp->phy_addr, PHY_82555_LED_NORMAL_CONTROL);

  spin_unlock_bh(&bdp->mdi_access_lock);
}

static int write_blink_led_timer(struct file *file, const char *buffer,
                                 unsigned long count, void *data)
{
    bd_config_t  *bdp = (bd_config_t *) data;
    char          s_blink_op[WRITE_BUF_MAX_LEN + 1];
    char         *res;
    unsigned long i_blink_op;
    
    if (!buffer)
    	return -EINVAL;
    
    if (count > WRITE_BUF_MAX_LEN) {
        count = WRITE_BUF_MAX_LEN;
    }
    
    copy_from_user(s_blink_op, buffer, count);
    s_blink_op[count] = '\0';
    i_blink_op = simple_strtoul(s_blink_op, &res, 0);
    if (res == s_blink_op) {
        return -EINVAL;
    }
    
    switch(i_blink_op){
     
        case LED_OFF:
	  set_led(bdp, PHY_82555_LED_OFF);
	    break;
        case LED_ON: 
            if(bdp->rev_id >= D101MA_REV_ID)
	      set_led(bdp, PHY_82555_LED_ON_559);
            else
	      set_led(bdp, PHY_82555_LED_ON_PRE_559);
	    
            break;
        default:
            return -EINVAL;
        }
    return count;
}


static e100_proc_entry e100_proc_list[] = {
    {"Description", read_descr, 0, 0},
    {"Driver_Name", read_drvr_name, 0, 0},
    {"Driver_Version", read_drvr_ver, 0, 0},
    {"PCI_Vendor", read_pci_vendor, 0, 0},
    {"PCI_Device_ID", read_pci_device, 0, 0},
    {"PCI_Subsystem_Vendor", read_pci_sub_vendor, 0, 0},
    {"PCI_Subsystem_ID", read_pci_sub_device, 0, 0},
    {"PCI_Revision_ID", read_pci_revision, 0, 0},
    {"PCI_Bus", read_pci_bus, 0, 0},
    {"PCI_Slot", read_pci_slot, 0, 0},
    {"IRQ", read_irq, 0, 0},
    {"System_Device_Name", read_dev_name, 0, 0},
    {"Current_HWaddr", read_current_hwaddr, 0, 0},
    {"Permanent_HWaddr", read_permanent_hwaddr, 0, 0},
    {"Part_Number", read_part_number, 0, PROC_LF},
    {"Link", read_link_status, 0, 0},
    {"Speed", read_speed, 0, 0},
    {"Duplex", read_dplx_mode, 0, 0},
    {"State", read_state, 0, PROC_LF},
    {"Rx_Packets", read_rx_packets, 0, 0},
    {"Tx_Packets", read_tx_packets, 0, 0},
    {"Rx_Bytes", read_rx_bytes, 0, 0},
    {"Tx_Bytes", read_tx_bytes, 0, 0},
    {"Rx_Errors", read_rx_errors, 0, 0},
    {"Tx_Errors", read_tx_errors, 0, 0},
    {"Rx_Dropped", read_rx_dropped, 0, 0},
    {"Tx_Dropped", read_tx_dropped, 0, 0},
    {"Multicast", read_rx_multicast_packets, 0, 0},
    {"Collisions", read_collisions, 0, 0},
    {"Rx_Length_Errors", read_rx_length_errors, 0, 0},
    {"Rx_Over_Errors", read_rx_over_errors, 0, 0},
    {"Rx_CRC_Errors", read_rx_crc_errors, 0, 0},
    {"Rx_Frame_Errors", read_rx_frame_errors, 0, 0},
    {"Rx_FIFO_Errors", read_rx_fifo_errors, 0, 0},
    {"Rx_Missed_Errors", read_rx_missed_errors, 0, 0},
    {"Tx_Aborted_Errors", read_tx_aborted_errors, 0, 0},
    {"Tx_Carrier_Errors", read_tx_carrier_errors, 0, 0},
    {"Tx_FIFO_Errors", read_tx_fifo_errors, 0, 0},
    {"Tx_Heartbeat_Errors", read_generic_0, 0, 0},
    {"Tx_Window_Errors", read_generic_0, 0, PROC_LF},
    {"Rx_TCP_Checksum_Good", read_generic_0, 0, 0},
    {"Rx_TCP_Checksum_Bad", read_generic_0, 0, 0},
    {"Tx_TCP_Checksum_Good", read_generic_0, 0, 0},
    {"Tx_TCP_Checksum_Bad", read_generic_0, 0, PROC_LF},
    {"Tx_Abort_Late_Coll", read_tx_abort_late_coll, 0, 0},
    {"Tx_Deferred_Ok", read_tx_deferred_ok, 0, 0},
    {"Tx_Single_Coll_Ok", read_tx_single_coll_ok, 0, 0},
    {"Tx_Multi_Coll_Ok", read_tx_multi_coll_ok, 0, 0},
    {"Rx_Long_Length_Errors", read_generic_0, 0, 0},
    {"Rx_Align_Errors", read_rx_align_errors, 0, PROC_LF},
    {"Tx_Flow_Control_Pause", read_tx_fc_packets, 0, 0},
    {"Rx_Flow_Control_Pause", read_rx_fc_packets, 0, 0},
    {"Rx_Flow_Control_Unsup", read_rx_fc_unsupported, 0, PROC_LF},
    {"Tx_TCO_Packets", read_tx_tco_packets, 0, 0},
    {"Rx_TCO_Packets", read_rx_tco_packets, 0, 0},
    {"Identify_Adapter", 0, write_blink_led_timer, 0},
    {"", 0, 0, 0}
};



/* need to check page boundaries !!! */
static int read_info(char *page, char **start, off_t off,
                     int count, int *eof, void *data)
{
    bd_config_t     *bdp = (bd_config_t *) data;
    char            *pagep = page;
    e100_proc_entry *pe;
    int              tmp;
    
    for (pe = e100_proc_list; pe->name[0]; pe++) {
        if (pe->read_proc) {
            
            if ((page + READ_BUF_MAX_LEN + E100_PE_LEN + 1) >= (pagep + PAGE_SIZE))
                break;
            
            page += sprintf(page, "%-" __MODULE_STRING(E100_PE_LEN) "s ", pe->name);
            page += pe->read_proc(page, start, 0, READ_BUF_MAX_LEN + 1, &tmp, bdp);
            
            if (pe->flags & PROC_LF)
                page += sprintf(page, "\n");
        }
    }
    
    *page = 0; 
    return generic_read(pagep, start, off, count, eof);
}

#ifdef E100_EOU
/**********************
 *  parameter entries
 **********************/
static void read_int_param(char *page, char *name, char *desc,
                           int def, int min, int max)
{
    page += sprintf(page, "Name: %s\n", name);
    page += sprintf(page, "Description: %s\n", desc);
    page += sprintf(page, "Default_Value: %d\n", def);
    page += sprintf(page, "Type: Range\n");
    page += sprintf(page, "Min: %d\n", min);
    page += sprintf(page, "Max: %d\n", max);
    page += sprintf(page, "Step:1\n");
    page += sprintf(page, "Radix: dec\n");
}

static void read_bool_param(char *page, char *name, char *desc, int def)
{
    page += sprintf(page, "Name: %s\n", name);
    page += sprintf(page, "Description: %s\n", desc);
    page += sprintf(page, "Default_Value: %d\n", def);
    page += sprintf(page, "Type: Enum\n");
    page += sprintf(page, "0: Off\n");
    page += sprintf(page, "1: On\n");
}

static int read_speed_duplex_def (char *page, char **start, off_t off,
                                  int count, int *eof, void *data)
{
    char *pagep = page;
    
    page += sprintf(page, "Name: Speed and Duplex\n");
    page += sprintf(page, "Description: Sets the adapter's speed and duplex mode\n");
    page += sprintf(page, "Default_Value: 0\n");
    page += sprintf(page, "Type: Enum\n");
    page += sprintf(page, "0: Auto-Negotiate\n");
    page += sprintf(page, "1: 10 Mbps / Half Duplex\n");
    page += sprintf(page, "2: 10 Mbps / Full Duplex\n");
    page += sprintf(page, "3: 100 Mbps / Half Duplex\n");
    page += sprintf(page, "4: 100 Mbps / Full Duplex\n");

    return generic_read(pagep, start, off, count, eof);
}

static int read_speed_duplex_val (char *page, char **start, off_t off,
                                  int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long)e100_speed_duplex[bdp->bd_number]);
}

static int read_tx_desc_def (char *page, char **start, off_t off,
                             int count, int *eof, void *data)
{
    read_int_param(page, "Transmit Descriptors",
                   "Sets the number of Tx descriptors available for the adapter",
                   E100_DEFAULT_TCB, E100_MIN_TCB, E100_MAX_TCB);

    return generic_read(page, start, off, count, eof);
}

static int read_tx_desc_val (char *page, char **start, off_t off,
                             int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long)TxDescriptors[bdp->bd_number]);
}

static int read_rx_desc_def (char *page, char **start, off_t off,
                             int count, int *eof, void *data)
{
    read_int_param(page, "Receive Descriptors",
                   "Sets the number of Rx descriptors available for the adapter",
                   E100_DEFAULT_RFD, E100_MIN_RFD, E100_MAX_RFD);

    return generic_read(page, start, off, count, eof);
}

static int read_rx_desc_val (char *page, char **start, off_t off,
                             int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long)RxDescriptors[bdp->bd_number]);
}

static int read_ber_def(char *page, char **start, off_t off,
                        int count, int *eof, void *data)
{
    read_int_param(page, "Bit Error Rate",
                   "Sets the value for the BER correction algorithm",
                   E100_DEFAULT_BER, 0, ZLOCK_MAX_ERRORS);

    return generic_read(page, start, off, count, eof);
}

static int read_ber_val(char *page, char **start, off_t off,
                        int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long)ber[bdp->bd_number]);
}

static int read_xsum_rx_def(char *page, char **start, off_t off,
                            int count, int *eof, void *data)
{
    read_bool_param(page, "RX Checksum",
                    "Setting this value to \"On\" enables receive checksum",
                    E100_DEFAULT_XSUM);
    
    return generic_read(page, start, off, count, eof);
}

static int read_xsum_rx_val(char *page, char **start, off_t off,
                            int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long)XsumRX[bdp->bd_number]);
}

static int read_ucode_def(char *page, char **start, off_t off,
                          int count, int *eof, void *data)
{
    read_bool_param(page, "Microcode",
                    "Setting this value to \"On\" enables the adapter's microcode",
                    E100_DEFAULT_UCODE);

    return generic_read(page, start, off, count, eof);
}

static int read_ucode_val(char *page, char **start, off_t off,
                          int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long)ucode[bdp->bd_number]);
}
static int read_bundle_small_def(char *page, char **start, off_t off,
                          int count, int *eof, void *data)
{
    read_bool_param(page, "Bundle Small Frames",
                    "Setting this value to \"On\" enables interrupt bundling of small frames",
                    E100_DEFAULT_BUNDLE_SMALL_FR);

    return generic_read(page, start, off, count, eof);
}

static int read_bundle_small_val(char *page, char **start, off_t off,
                          int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long)BundleSmallFr[bdp->bd_number]);
}

static int read_fc_def(char *page, char **start, off_t off,
                       int count, int *eof, void *data)
{
    read_bool_param(page, "Flow Control",
                    "Setting this value to \"On\" enables processing flow-control packets",
                    E100_DEFAULT_FC);

    return generic_read(page, start, off, count, eof);
}

static int read_fc_val(char *page, char **start, off_t off,
                       int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long)flow_control[bdp->bd_number]);
}

static int read_int_delay_def(char *page, char **start, off_t off,
                              int count, int *eof, void *data)
{
    read_int_param(page, "CPU Saver Interrupt Delay",
                   "Sets the value for CPU saver's interrupt delay",
                   E100_DEFAULT_CPUSAVER_INTERRUPT_DELAY, 0x0, 0xFFFF);

    return generic_read(page, start, off, count, eof);
}

static int read_int_delay_val(char *page, char **start, off_t off,
                              int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long)IntDelay[bdp->bd_number]);
}

static int read_bundle_max_def(char *page, char **start, off_t off,
                               int count, int *eof, void *data)
{
    read_int_param(page, "CPU Saver Maximum Bundle",
                   "Sets the value for CPU saver's maximum value",
                   E100_DEFAULT_CPUSAVER_BUNDLE_MAX, 0x1, 0xFFFF);

    return generic_read(page, start, off, count, eof);
}

static int read_bundle_max_val(char *page, char **start, off_t off,
                               int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long)BundleMax[bdp->bd_number]);
}

static int read_ifs_def(char *page, char **start, off_t off,
                        int count, int *eof, void *data)
{
    read_bool_param(page, "IFS",
                    "Setting this value to \"On\" enables the adaptive IFS algorithm",
                    E100_DEFAULT_IFS);

    return generic_read(page, start, off, count, eof);
}

static int read_ifs_val(char *page, char **start, off_t off,
                        int count, int *eof, void *data)
{
    bd_config_t *bdp = (bd_config_t *) data;

    return read_ulong(page, start, off, count, eof, 
                      (unsigned long)IFS[bdp->bd_number]);
}

static e100_proc_entry e100_proc_params[] = {
    {"e100_speed_duplex.def", read_speed_duplex_def, 0, 0},
    {"e100_speed_duplex.val", read_speed_duplex_val, 0, 0},
    {"RxDescriptors.def", read_rx_desc_def, 0, 0},
    {"RxDescriptors.val", read_rx_desc_val, 0, 0},
    {"TxDescriptors.def", read_tx_desc_def, 0, 0},
    {"TxDescriptors.val", read_tx_desc_val, 0, 0},
    {"XsumRX.def", read_xsum_rx_def, 0, 0},
    {"XsumRX.val", read_xsum_rx_val, 0, 0},
    {"ucode.def", read_ucode_def, 0, 0},
    {"ucode.val", read_ucode_val, 0, 0},
    {"BundleSmallFr.def", read_bundle_small_def, 0, 0},
    {"BundleSmallFr.val", read_bundle_small_val, 0, 0},
    {"IntDelay.def", read_int_delay_def, 0, 0},
    {"IntDelay.val", read_int_delay_val, 0, 0},
    {"BundleMax.def", read_bundle_max_def, 0, 0},
    {"BundleMax.val", read_bundle_max_val, 0, 0},
    {"ber.def", read_ber_def, 0, 0},
    {"ber.val", read_ber_val, 0, 0},
    {"flow_control.def", read_fc_def, 0, 0},
    {"flow_control.val", read_fc_val, 0, 0},
    {"IFS.def", read_ifs_def, 0, 0},
    {"IFS.val", read_ifs_val, 0, 0},
    {"", 0, 0, 0}
};
#endif /* E100_EOU */

static struct proc_dir_entry *create_proc_rw(char *name, bd_config_t *bdp,
                                             struct proc_dir_entry *parent,
                                             read_proc_t *read_proc,
                                             write_proc_t *write_proc)
{
    struct proc_dir_entry *pdep;
    mode_t mode = S_IFREG;

    if (write_proc) {
    	mode |= S_IWUSR;
        if (read_proc) {
            mode |= S_IRUSR;
        }
    
    } else if (read_proc) {
    	mode |= S_IRUGO;
    }
    
    if (!(pdep = create_proc_entry(name, mode , parent)))
        return NULL;
    
    pdep->read_proc = read_proc;
    pdep->write_proc = write_proc;
    pdep->data = bdp;
    return pdep;
}

#ifdef E100_EOU
static int create_proc_param_subdir(bd_config_t *bdp, struct proc_dir_entry *dev_dir)
{
    struct proc_dir_entry *param_dir;
    e100_proc_entry       *pe;
    
    param_dir = create_proc_entry("LoadParameters", S_IFDIR, dev_dir);
    if (!param_dir)
        return -1;
        
    for (pe = e100_proc_params; pe->name[0]; pe++) {
        if (!(create_proc_rw(pe->name, bdp, param_dir, pe->read_proc, pe->write_proc))) {
            e100_remove_proc_subdir(bdp);
            return -1;
        }
    }
    
    return 1;
}

static void remove_proc_param_subdir(struct proc_dir_entry *parent)
{
    struct proc_dir_entry *de;
    e100_proc_entry       *pe;
    int                    len;

    len = strlen("LoadParameters");
    
    for (de = parent->subdir; de; de = de->next) {
        if ((de->namelen == len) && (!memcmp(de->name, "LoadParameters", len)))
            break;
    }
    
    if (!de)
        return;
    
    for (pe = e100_proc_params; pe->name[0]; pe++) {
        remove_proc_entry(pe->name, de);
    }
    
    remove_proc_entry("LoadParameters", parent);
}
#endif /* E100_EOU */

int e100_create_proc_subdir(bd_config_t * bdp)
{
    struct proc_dir_entry *dev_dir;
    e100_proc_entry       *pe;
    char                   info[256];
    int                    len;
    
    /* create the main /proc dir if needed */
    if (!adapters_proc_dir) {
        if (e100_init_proc_dir() != B_TRUE)
            return -1;
    }
    
    strncpy(info, bdp->device->name, sizeof(info));
    len = strlen(info);
    strncat(info + len, ".info", sizeof(info) - len);
    
    /* info */
    if (!(create_proc_rw(info, bdp, adapters_proc_dir, read_info, 0))) {
        e100_proc_cleanup();
        return -1;
    }

    dev_dir = create_proc_entry(bdp->device->name, S_IFDIR, adapters_proc_dir);
    bdp->proc_parent = dev_dir;
    
    if (!dev_dir) {
        e100_remove_proc_subdir(bdp);
        return -1;
    }
    
    for (pe = e100_proc_list; pe->name[0]; pe++) {
        if (!(create_proc_rw(pe->name, bdp, dev_dir, pe->read_proc, pe->write_proc))) {
            e100_remove_proc_subdir(bdp);
            return -1;
        }
    }
    
#ifdef E100_EOU
    if (create_proc_param_subdir(bdp, dev_dir) == -1) {
        e100_remove_proc_subdir(bdp);
        return -1;
    }
#endif
    
    return 0;
}

void e100_remove_proc_subdir(bd_config_t * bdp)
{
    e100_proc_entry *pe;
    char             info[256];
    int              len;

    len = strlen(bdp->device->name);
    strncpy(info, bdp->device->name, sizeof(info));
    strncat(info + len, ".info", sizeof(info) - len);

    if (bdp->proc_parent) {
        for (pe = e100_proc_list; pe->name[0]; pe++) {
            remove_proc_entry(pe->name, bdp->proc_parent);
        }
        
#ifdef E100_EOU
        remove_proc_param_subdir(bdp->proc_parent);
#endif
        remove_proc_entry(bdp->device->name, adapters_proc_dir);
    }

    remove_proc_entry(info, adapters_proc_dir);
    
    /* try to remove the main /proc dir, if it's empty */
    e100_proc_cleanup();
}


/****************************************************************************
 * Name:          e100_init_proc_dir
 *
 * Description:   This routine creates the top-level /proc directory for the
 *                driver in /proc/net
 *
 * Arguments:     none
 *
 * Returns:       B_TRUE on success, B_FALSE on fail
 *
 ***************************************************************************/
static unsigned char e100_init_proc_dir(void)
{
    int len;
    
    /* first check if adapters_proc_dir already exists */
    len = strlen(ADAPTERS_PROC_DIR);
    for (adapters_proc_dir = proc_net->subdir;
         adapters_proc_dir;
         adapters_proc_dir = adapters_proc_dir->next) {
        
        if ((adapters_proc_dir->namelen == len) && 
            (!memcmp(adapters_proc_dir->name, ADAPTERS_PROC_DIR, len)))
            break;
    }
    
    if (!adapters_proc_dir)     
        adapters_proc_dir = create_proc_entry(ADAPTERS_PROC_DIR, S_IFDIR, proc_net);

    if (!adapters_proc_dir)
        return B_FALSE;

    return B_TRUE;
}

/****************************************************************************
 * Name:          e100_proc_cleanup
 *
 * Description:   This routine clears the top-level /proc directory, if empty.
 *
 * Arguments:     none
 *
 * Returns:       none
 *
 ***************************************************************************/
static void e100_proc_cleanup(void)
{
    struct proc_dir_entry   *de;

    /* sanity check, due to a kernel bug in proc/generic.c */
    if (adapters_proc_dir == NULL) {
        return;
    }
    
    /* check if the subdir list is empty before removing adapters_proc_dir */
    for (de = adapters_proc_dir->subdir; de; de = de->next) {
        /* ignore . and .. */
        if (*(de->name) != '.')
            break;
    }
    
    if (de)
        return;

    remove_proc_entry(ADAPTERS_PROC_DIR, proc_net);
    adapters_proc_dir = NULL;
}

#endif /* CONFIG_PROC_FS */
